use std::{fmt::Debug, fmt::Display, str::FromStr};

use dialoguer::{theme::ColorfulTheme, Confirm, Input, Select, Validator};
use std::io::Result;
use url::Url;

pub fn select<'a, T: ToString>(prompt: &str, items: &'a [T], opt: bool) -> Option<&'a T> {
    let index = select_index(prompt, items, opt).ok()??;
    Some(&items[index])
}

pub fn select_index<T: ToString>(prompt: &str, items: &[T], opt: bool) -> Result<Option<usize>> {
    let theme = &ColorfulTheme::default();
    let mut select = Select::with_theme(theme);
    select.with_prompt(prompt).default(0).items(items);
    if opt {
        select.interact_opt()
    } else {
        Ok(Some(select.interact()?))
    }
}

pub fn confirm(prompt: &str, default: Option<bool>) -> Result<bool> {
    let theme = ColorfulTheme::default();
    let mut confirm = Confirm::with_theme(&theme);
    confirm.with_prompt(prompt);
    if let Some(x) = default {
        confirm.default(x);
    }
    confirm.interact()
}

#[cfg(unix)]
pub fn unix_only_prompt(message: &str) -> Result<bool> {
    Ok(confirm(message, Some(false))?)
}

#[cfg(not(unix))]
pub fn unix_only_prompt(_: &str) -> Result<bool> {
    Ok(false)
}

pub fn input<T>(prompt: &str) -> Result<T>
where
    T: Clone + FromStr + Display,
    T::Err: Display + Debug,
{
    Input::with_theme(&ColorfulTheme::default())
        .with_prompt(prompt)
        .interact_text()
}

pub fn input_validate<T, F>(prompt: &str, validator: F) -> Result<T>
where
    T: Clone + FromStr + Display,
    T::Err: Display + Debug,
    F: Validator<T>,
{
    Input::with_theme(&ColorfulTheme::default())
        .with_prompt(prompt)
        .validate_with(validator)
        .interact_text()
}

pub fn input_url(prompt: &str) -> Result<Url> {
    let url = input_validate(prompt, |input: &String| -> core::result::Result<(), &str> {
        match Url::parse(input) {
            Ok(_) => Ok(()),
            Err(_) => Err("This is not a valid url"),
        }
    })?;
    Ok(Url::parse(&url).unwrap())
}
