mod config;
mod hosts;
mod ui;

use anyhow::Result;
use clap::{clap_app, crate_version};
use config::Repo;
use config::Upstream;
use config::{Artifact, Host, HostDiscriminants};
use config::{FromPrompt, NoPkgError};
use console::style;
use hosts::github::GitHub;
use indicatif::{ProgressBar, ProgressStyle};
use regex::Regex;
use std::{
    fs::{self, File},
    io::{Read, Write},
};
use std::{path::Path, str::FromStr};
use strum::VariantNames;
use url::Url;

fn main() {
    let matches = clap_app!(nopkg =>
        (version: crate_version!())
        (@subcommand new =>
            (about: "Configure a new repo")
        )
        (@subcommand list =>
            (about: "List configured repos")
        )
        (@subcommand delete =>
            (about: "Delete a repo")
        )
        (@subcommand pull =>
            (about: "Pull artifacts from repo")
        )
        (@setting SubcommandRequiredElseHelp)
    )
    .get_matches();

    let result = match matches.subcommand.unwrap().name.as_ref() {
        "new" => {
            start_ui();
            new()
        }
        "list" => list(),
        "delete" => delete(),
        "pull" => pull(),
        _ => panic!("Command not found and help command failed to run"),
    };

    match result {
        Ok(_) => (),
        Err(why) => eprintln!(
            "{} {} {}",
            style("Error").bold().red(),
            style("-").bold(),
            style(why).bold()
        ),
    };
}

//// Subcommand handlers
fn new() -> Result<()> {
    let empty: Result<Vec<Upstream>> = Ok(Vec::new());
    let mut config: Vec<Upstream> = get_config().or(empty)?;
    let upstream: Upstream = new_upstream()?;
    config.push(upstream);
    pull_from_upstream(&mut config)?;
    write_config(&config)?;
    Ok(())
}

fn list() -> Result<()> {
    let config: Vec<Upstream> = get_config()?;
    for (index, upstream) in config.iter().enumerate() {
        if index != 0 {
            println!(); // Add line between hosts
        };
        println!("{}", upstream);
    }
    Ok(())
}

fn delete() -> Result<()> {
    let mut config: Vec<Upstream> = get_config()?;
    let index = ui::select_index(
        "Delete which host?",
        &config
            .iter()
            .map(|upstream| upstream.host.get_display())
            .collect::<Vec<String>>(),
        true,
    )?;
    match index {
        Some(x) => config.remove(x),
        None => return Ok(()),
    };
    write_config(&config)?;
    Ok(())
}

fn pull() -> Result<()> {
    let mut config: Vec<Upstream> = get_config()?;
    pull_from_upstream(&mut config)?;
    write_config(&config)?;
    Ok(())
}

//// Helper commands
fn get_config() -> Result<Vec<Upstream>> {
    let file = match File::open(".nopkg") {
        Ok(x) => x,
        Err(err) => match err.kind() {
            std::io::ErrorKind::NotFound => return Err(NoPkgError::NoConfig.into()),
            _ => return Err(err.into()),
        },
    };
    match bincode::deserialize_from(&file).ok() {
        Some(x) => Ok(x),
        None => Ok(Vec::new()),
    }
}

fn write_config(config: &Vec<Upstream>) -> Result<()> {
    let file = File::create(".nopkg")?;
    bincode::serialize_into(file, &config)?;
    Ok(())
}

fn pull_from_upstream(config: &mut Vec<Upstream>) -> Result<()> {
    for mut upstream in config {
        let host: &Host = (&upstream.host).into();

        let spinner = ProgressBar::new_spinner();
        spinner.set_message(&format!(
            "{} {}",
            style("Checking for updates for").bold(),
            style(host.get_display()).blue().bold()
        ));
        spinner.enable_steady_tick(150);

        let to_upgrade = match host.should_upgrade(&upstream.version) {
            Ok(x) => x,
            Err(why) => {
                spinner
                    .with_style(ProgressStyle::default_spinner().template("{msg}"))
                    .finish_with_message(&format!(
                        "{} {}",
                        style("✗").red().bold(),
                        style(why).bold(),
                    ));
                continue;
            }
        };

        let mut force_redownload = false;
        for file in &upstream.entries {
            if !Path::new(&file.name).exists() {
                force_redownload = true;
            }
        }

        if to_upgrade.0 || force_redownload {
            spinner.finish_and_clear();
            println!(
                "{} {} {} {} {}",
                style("»").bold().blue(),
                style("Upgrading").bold(),
                style(host.get_display()).blue().bold(),
                style("to version").bold(),
                style(&to_upgrade.1).bold()
            );
        } else {
            spinner
                .with_style(ProgressStyle::default_spinner().template("{msg}"))
                .finish_with_message(&format!(
                    "{} {} {}",
                    style("✔").bold().green(),
                    style(host.get_display()).bold().blue(),
                    style("up to date!").bold(),
                ));
            continue;
        }
        let urls = host.pull(&upstream.entries)?;
        for (entry, url) in upstream.entries.iter().zip(urls) {
            download_file(&entry, &url)?;
        }
        upstream.version = to_upgrade.1; // update to new version
    }
    Ok(())
}

#[cfg(unix)]
pub fn set_bit(file: File) -> Result<()> {
    use std::os::unix::fs::PermissionsExt;
    let mut perms = file.metadata()?.permissions();
    // if bit already set then return
    if perms.mode() % 2 != 0 {
        return Ok(());
    }
    perms.set_mode(perms.mode() + 0o100); // Invert executable bit for owner
    file.set_permissions(perms)?;
    Ok(())
}

#[cfg(not(unix))]
pub fn set_bit(_: File) -> Result<()> {
    Ok(())
}

fn download_file(entry: &Artifact, url: &Url) -> Result<()> {
    let spinner = ProgressBar::new_spinner();
    spinner.set_style(ProgressStyle::default_spinner().template("    {spinner} {msg}"));
    spinner.enable_steady_tick(150);
    spinner.set_message(&format!(
        "{} {}",
        style("Downloading file").bold(),
        entry.name
    ));
    let mut reader = ureq::get(&url.to_string()).call().into_reader();
    let mut bytes = vec![];
    reader.read_to_end(&mut bytes)?;
    let path = Path::new(&entry.name);
    if path.exists() {
        fs::remove_file(path)?;
    };
    let mut file = File::create(&path)?;
    file.write(&bytes)?;
    if entry.executable {
        set_bit(file)?;
    }

    spinner
        .with_style(ProgressStyle::default_spinner().template("    {msg}"))
        .finish_with_message(&format!(
            "{} {} {}",
            style("✔").bold().green(),
            style("Downloaded file").bold(),
            &entry.name
        ));
    Ok(())
}

// Make sure cursor shows up if user exits multi-select menu with Ctrl+C
fn start_ui() {
    ctrlc::set_handler(move || {
        let term = console::Term::stdout();
        let _ = term.show_cursor();
    })
    .expect("Failed to set up Ctrl+C handler");
}

fn new_entry() -> Result<Artifact> {
    let regex: String = ui::input_validate(
        "Regex of file to pull",
        |input: &String| -> Result<(), &str> {
            match Regex::new(input) {
                Ok(_) => Ok(()),
                Err(_) => Err("This is not a valid regex"),
            }
        },
    )?;
    let name: String = ui::input("Output filename")?;
    let executable: bool = ui::unix_only_prompt("Set executable bit?")?;
    Ok(Artifact {
        name,
        regex,
        executable,
    })
}

fn new_upstream() -> Result<Upstream> {
    let host_str = match ui::select("Select the host", &HostDiscriminants::VARIANTS, false) {
        Some(x) => x,
        None => return Err(NoPkgError::UiError("No host selected".to_string()).into()),
    };
    let host: Host =
        match HostDiscriminants::from_str(host_str).expect("Unable to decode selected host") {
            HostDiscriminants::GitHub => Host::GitHub(GitHub::from_prompt()?),
            // HostDiscriminants::Gitea => Host::Gitea(Gitea::from_prompt()?),
        };
    let mut entries: Vec<Artifact> = Vec::new();
    loop {
        entries.push(new_entry()?);
        if !ui::confirm("Add another file?", None)? {
            break;
        }
    }
    Ok(Upstream {
        host,
        entries,
        version: "".to_owned(),
    })
}
