use crate::config::{Artifact, NoPkgError};
use crate::config::{FromPrompt, Repo};
use crate::ui;
use anyhow::Result;
use chrono::{DateTime, Utc};
use regex::Regex;
use serde::{Deserialize, Serialize};
use url::Url;

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct GitHub {
    domain: String,
    user: String,
    repo: String,
    prerelease: bool,
}

#[derive(Serialize, Deserialize, Debug)]
struct Asset {
    #[serde(rename = "browser_download_url")]
    url: String,
    name: String,
}

#[derive(Serialize, Deserialize, Debug)]
struct Release {
    tag_name: String,
    assets: Vec<Asset>,
    prerelease: bool,
    published_at: DateTime<Utc>,
}

impl FromPrompt for GitHub {
    fn from_prompt() -> Result<Self, std::io::Error> {
        let do_domain: bool = ui::confirm("Use a GitHub enterprise domain?", Some(false))?;
        let domain: String = if do_domain {
            ui::input_url("Enterprise domain URL")?.into_string()
        } else {
            String::from("https://api.github.com/") // trailing slash since that's what Url standardizes to
        };
        let prerelease: bool = ui::confirm("Pull from prereleases?", Some(false))?;
        let user: String = ui::input("Repository user")?;
        let repo: String = ui::input("Repository slug")?;
        Ok(GitHub {
            domain,
            user,
            repo,
            prerelease,
        })
    }
}

impl GitHub {
    fn get_releases(&self) -> Result<Vec<Release>> {
        let url = format!("{}repos/{}/{}/releases", self.domain, self.user, self.repo);
        let out = ureq::get(&url).call().into_string()?;
        if out.contains("API rate limit exceeded") {
            return Err(NoPkgError::PullError("Encountered rate limit".to_string()).into());
        };
        let releases: Vec<Release> = serde_json::from_str(&out)?;
        Ok(releases)
    }
}

impl Repo for GitHub {
    fn should_upgrade(&self, old: &str) -> Result<(bool, String)> {
        let mut releases = self.get_releases()?;
        releases.sort_unstable_by(|a, b| b.published_at.partial_cmp(&a.published_at).unwrap());
        let releases: Vec<&Release> = releases
            .iter()
            .filter(|x| !self.prerelease && !x.prerelease)
            .collect();
        let version: &String = &releases[0].tag_name;
        if version != old {
            Ok((true, version.to_owned()))
        } else {
            Ok((false, old.to_owned()))
        }
    }

    fn pull(&self, artifacts: &Vec<Artifact>) -> Result<Vec<Url>> {
        let releases = self.get_releases()?;
        let mut urls: Vec<Url> = Vec::new();
        for artifact in artifacts {
            let regex = Regex::new(&format!("^{}$", &artifact.regex))?;
            let matched: Vec<&Asset> = releases[0]
                .assets
                .iter()
                .filter(|x| regex.is_match(&x.name))
                .collect();
            if matched.len() > 1 {
                return Err(NoPkgError::PullError(format!(
                    "More than one artifact matched by regex {}",
                    regex
                ))
                .into());
            }
            if matched.len() == 0 {
                return Err(NoPkgError::PullError(format!(
                    "No artifact matched by regex {}",
                    regex
                ))
                .into());
            }
            urls.push(Url::parse(&matched[0].url)?);
        }
        assert_eq!(artifacts.len(), urls.len());
        Ok(urls)
    }

    fn get_display(&self) -> String {
        format!("{}/{}", self.user, self.repo)
    }
}
