use console::style;
use std::fmt::{Debug, Display};
use thiserror::Error;

use crate::hosts::github::GitHub;
use anyhow::Result;
use enum_dispatch::enum_dispatch;
use serde::{Deserialize, Serialize};
use strum::{Display, EnumDiscriminants, EnumString, EnumVariantNames};
use url::Url;

#[derive(Debug, Error)]
pub enum NoPkgError {
    #[error("Failure while pulling artifacts: {0}")]
    PullError(String),
    #[error("Invalid input: {0}")]
    UiError(String),
    #[error("No config available, use the new subcommand to create one")]
    NoConfig,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct Upstream {
    pub host: Host,
    pub entries: Vec<Artifact>,
    pub version: String, // most recently pulled version
}

impl Display for Upstream {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "{}: {}\n",
            style("Host").bold(),
            style(&self.host).blue().bold()
        )?;
        write!(
            f,
            "{}: {}\n",
            style("Name").bold(),
            style(&self.host.get_display()).blue().bold()
        )?;
        write!(
            f,
            "{}: {}\n",
            style("Current Version").bold(),
            style(&self.version).blue().bold()
        )?;
        write!(f, "{}:", style("Files").bold())?;
        for entry in &self.entries {
            write!(f, "\n{}", &entry)?;
        }
        Ok(())
    }
}

#[enum_dispatch]
#[derive(Serialize, Deserialize, EnumDiscriminants, Debug, Clone, Display)]
#[strum_discriminants(derive(EnumVariantNames, EnumString))]
pub enum Host {
    GitHub(GitHub),
}

#[derive(Serialize, Deserialize, Debug)]
pub struct Artifact {
    pub name: String,
    pub regex: String,
    pub executable: bool, // whether to set executable bit
}

impl Display for Artifact {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "    {}: {}\n",
            style("Filename").bold(),
            style(&self.name).green().bold()
        )?;
        write!(
            f,
            "    {}: {}",
            style("Regex").bold(),
            style(&self.regex).green().bold()
        )?;
        Ok(())
    }
}

pub trait FromPrompt {
    fn from_prompt() -> Result<Self, std::io::Error>
    where
        Self: Sized;
}

#[enum_dispatch(Host)]
pub trait Repo {
    fn should_upgrade(&self, current: &str) -> Result<(bool, String)>;
    fn pull(&self, artifacts: &Vec<Artifact>) -> Result<Vec<Url>>;
    fn get_display(&self) -> String;
}
