//! Builders for messages and other protocol elements.
//!
//! These builders write protocol elements directly into the given packet buffer where possible.
//!
//! Each protocol element defines a [`Builder`], which contains element-specific state and owns a
//! parent [`Builder`] that it will release once built. The root of this hierarchy is [`Sink`],
//! which owns the actual packet buffer.
//!
//! DNS has many variable-length protocol elements, so to reconcile this with our zero-copy
//! approach, the methods on the [`Builder`] types form a [state machine].
//!
//! # Limitations
//!
//! [`Builder`] is public, so external code can name the necessary trait bounds to pass and return
//! builders across function boundaries, but only internal code can implement builders that directly
//! interact with the [`Sink`]. This is unfortunate but necessary to preserve the state machine’s
//! correctness guarantees.
//!
//! [state machine]: https://hoverbear.org/blog/rust-state-machine-pattern/

#![allow(unknown_lints)]
#![allow(private_intra_doc_links)]

#[cfg_attr(doctest, macro_export)]
macro_rules! builder {
    // @method_real helper generates a function item for an inherent method.
    {
        @method_real ($B:ident $P:ident)                    // @method_real (B P)
        ($builder:ident) ($($state:ident)?)                 // (FooBuilder) (FooBar)
        ($(#[$meta:meta])*) ($vis:vis) ($method:ident)      // (#[doc = "a"]) (pub) (handle_tu)
        ($($($fgeneric:tt)+)?) ($self:ident $($arg:tt)*)    // (T, U: Ord) (self, t: T, u: U)
        ($($ret:ty)?)                                       // (Self)
        ($($body:tt)*)                                      // (t; u; self)
    } => {
        $(#[$meta])*
        #[allow(unused_mut)]
        $vis fn $method $(<$($fgeneric)+>)? (mut $self $($arg)*) $(-> $ret)? { $($body)* }
    };
    // @method helper ({body} version) calls @method_real for methods with an explicit body.
    {
        @method ($B:ident $P:ident)                         // @method (B P)
        ($builder:ident) ($($state:ident)?)                 // (FooBuilder) (FooBar)
        $_:tt                                               // // unused
        $meta:tt $vis:tt $method:tt                         // (#[doc = "a"]) (pub) (handle_tu)
        $fgeneric:tt $args:tt                               // (T, U: Ord) (self, t: T, u: U)
        ($($ret:ty)?)                                       // (Self)
        ({$($body:tt)*})                                    // ({t; u; self})
    } => {
        builder! {
            @method_real ($B $P)
            ($builder) ($($state)?)
            $meta $vis $method
            $fgeneric $args
            ($($ret)?)
            ({$($body)*})
        }
    };
    // @method helper ([into] version) calls @method_real for Into-based transition methods.
    {
        @method ($B:ident $P:ident)                         // @method (B P)
        ($builder:ident) ($($state:ident)?)                 // (FooBuilder) (FooBar)
        ($parent:ty)                                        // (P)
        $meta:tt $vis:tt $method:tt                         // (#[doc = "a"]) (pub) (into_baz)
        $fgeneric:tt $args:tt                               // (T, U: Ord) (self, t: T, u: U)
        ()                                                  // () // no $ret allowed
        (                                                   // (
            [into $state2:tt]                               // [into FooBaz]
            {$($body:tt)*}                                  // { self.handle_tu(t, u) }
        )                                                   // )
    } => {
        builder! {
            @method_real ($B $P)
            ($builder) ($($state)?)
            $meta $vis $method
            $fgeneric $args
            ($builder<$B, $parent, $state2>)
            ({$($body)*}.into())
        }
    };
    // @method helper ([try_into] version) calls @method_real for TryInto-based transition methods.
    {
        @method ($B:ident $P:ident)                         // @method (B P)
        ($builder:ident) ($($state:ident)?)                 // (FooBuilder) (FooBar)
        ($parent:ty)                                        // (P)
        $meta:tt $vis:tt $method:tt                         // (#[doc = "a"]) (pub) (try_into_baz)
        $fgeneric:tt $args:tt                               // (T, U: Ord) (self, t: T, u: U)
        ()                                                  // () // no $ret allowed
        (                                                   // (
            [try_into $state2:tt | $etype:ident]            // [try_into FooBaz | FooError]
            {$($body:tt)*}                                  // { self.handle_tu(t, u) }
        )                                                   // )
    } => {
        builder! {
            @method_real ($B $P)
            ($builder) ($($state)?)
            $meta $vis $method
            $fgeneric $args
            (Result<$builder<$B, $parent, $state2>, $etype>)
            ({$($body)*}.try_into())
        }
    };
    // @method helper ([push] version) calls @method_real for PushBuilder convenience methods.
    {
        @method ($B:ident $P:ident)                         // @method (B P)
        ($builder:ident) ($($state:ident)?)                 // (FooBuilder) (FooBar)
        ($parent:ty)                                        // (P)
        $meta:tt $vis:tt $method:tt                         // (#[doc = "a"]) (pub) (push_child)
        $fgeneric:tt $args:tt                               // (T, U: Ord) (self, t: T, u: U)
        ()                                                  // () // no $ret allowed
        (                                                   // (
            [push $builder2:ident$(<$state2:tt>)?           // [push ChildBuilder<Baby>
            | $etype:ident::$evariant:ident]                // | FooError::Child]
            {$($body:tt)*}                                  // { self.handle_tu(t, u) }
        )                                                   // )
    } => {
        builder!{
            @method_real ($B $P)
            ($builder) ($($state)?)
            $meta $vis $method
            $fgeneric $args
            (Result<
                $builder2<$B, $builder<$B, $parent $(, $state)?> $(, $state2)?>,
                $etype
            >)
            ($builder2::push({$($body)*}).map_err($etype::$evariant))
        }
    };
    // @impl helper generates an inherent impl block and calls @method for each given $item.
    {
        @impl ($B:ident $P:ident)                           // @impl (B P)
        ($($bgeneric:tt)+)                                  // (B, P, Q) // HACK
        ($builder:ident) $state:tt                          // (FooBuilder) (Q)
        ($parent:ty)                                        // (P)
        ($($($igeneric:tt)+)?)                              // (Q: SpecialStates)
        $(($($item:tt)*))*                                  // ((#[doc = "a"]) (pub) ...) ...
    } => {
        impl<$B: Buffer, $P: Builder<$B> $(, $($igeneric)+)?>
        $builder<$($bgeneric)+> {
            $(builder! {
                @method ($B $P)
                ($builder) $state
                ($parent)
                $($item)*
            })*
        }
    };
    // Main rule consumes Rust-like builder definition syntax from the author.
    {
        <$B:ident, $P:ident> $builder:ident {
            Builder $([$($bgeneric:tt)+])?;
            $(@ <$parent:ty> $($state:ident)? $([$($igeneric:tt)+])?: $(
                $(#[$meta:meta])* $vis:vis fn $method:ident
                $([$($fgeneric:tt)+])? (mut $self:ident $($arg:tt)*)
                $(-> $ret:ty)? =
                $([$($special:tt)+])?
                {$($body:tt)*}
            )*)*
        }
    } => {
        impl<$B: Buffer, $P: Builder<$B> $(, $($bgeneric)+)?>
        Builder<$B> for $builder<$B, $P $(, $($bgeneric)+)?> {
            fn sink(&mut self) -> &mut Sink<B> {
                self.parent.sink()
            }
        }

        $(#[allow(unused)] builder! {
            @impl ($B $P)
            ($B, $parent $(, $state)?)
            ($builder) ($($state)?)
            ($parent)
            ($($($igeneric)+)?)
            $((
                ($(#[$meta])*) ($vis) ($method)
                ($($($fgeneric)+)?) ($self $($arg)*)
                ($($ret)?)
                ($([$($special)+])? {$($body)*})
            ))*
        })*
    };
}

macro_rules! transition {
    {$($builder:ident.$field:ident {$(($($extra:ident),*) $from:ident -> $to:ident;)*})*} => {$($(
        impl<B: Buffer, P: Builder<B>> From<$builder<B, P, $from>> for $builder<B, P, $to> {
            fn from(builder: $builder<B, P, $from>) -> Self {
                $builder {
                    $field: $to,
                    $($extra: builder.$extra,)*
                }
            }
        }
    )*)*};
}

#[cfg(test)]
pub mod example;
pub mod extension;
pub mod message;
pub mod name;
pub mod question;
pub mod record;

use core::convert::{TryFrom, TryInto};
use core::num::NonZeroU16;
use core::ops::{DerefMut, Range};

use arrayvec::{Array, ArrayVec};

#[cfg(feature = "alloc")]
use alloc::vec::Vec;

/// A mutable packet buffer.
pub trait Buffer: DerefMut<Target = [u8]> {
    fn resize_zero(&mut self, len: usize);
    fn capacity_limit(&self) -> Option<usize>;
}

/// The root builder.
pub struct Sink<B> {
    inner: B,
    limit: Option<NonZeroU16>,
}

error!(SinkError);
/// failed to create sink
#[derive(Debug, displaydoc::Display)]
#[prefix_enum_doc_attributes]
pub enum SinkError {
    /// inner must be empty (but had len = {0})
    InnerNotEmpty(usize),

    /// limit ({0:?}) exceeds capacity limit of inner ({1})
    LimitTooHigh(Option<NonZeroU16>, usize),

    /// capacity limit of inner ({0}) overflows u16
    CapacityLimitTooHigh(usize),
}

error!(GrowError);
/// failed to grow sink
#[derive(Debug, displaydoc::Display)]
#[prefix_enum_doc_attributes]
pub enum GrowError {
    /// old len ({0}) plus delta ({1}) overflows usize
    Overflow(usize, usize),

    /// old len ({0}) plus delta ({1}) exceeds limit ({2})
    Limit(usize, usize, NonZeroU16),
}

impl<B: Buffer> Sink<B> {
    pub fn with_limit(inner: B, limit: u16) -> Result<Self, SinkError> {
        let limit =
            NonZeroU16::new(limit.max(1u8.into())).expect("clamp to [1, ∞) makes this infallible");

        Self::new0(inner, Some(limit))
    }

    fn new0(inner: B, limit: Option<NonZeroU16>) -> Result<Self, SinkError> {
        if !inner.is_empty() {
            return Err(SinkError::InnerNotEmpty(inner.len()));
        }

        let minimum = NonZeroU16::new(512).expect("is not zero");
        let limit = limit.map(|x| x.max(minimum));

        match (limit, inner.capacity_limit()) {
            (None, Some(x)) => return Err(SinkError::LimitTooHigh(None, x)),
            (Some(y), Some(x)) if usize::from(u16::from(y)) > x => {
                return Err(SinkError::LimitTooHigh(Some(y), x))
            }
            _ => {}
        }

        Ok(Sink { inner, limit })
    }

    pub fn finish(self) -> B {
        self.inner
    }

    fn grow_range(&mut self, delta: usize) -> Result<Range<usize>, GrowError> {
        let start = self.inner().len();
        let stop = start
            .checked_add(delta)
            .ok_or_else(|| GrowError::Overflow(start, delta))?;

        if let Some(limit) = self.limit {
            if stop > usize::from(limit.get()) {
                return Err(GrowError::Limit(start, delta, limit));
            }
        }

        self.inner.resize_zero(stop);

        Ok(start..stop)
    }

    fn grow_mut(&mut self, delta: usize) -> Result<&mut [u8], GrowError> {
        let result = self.grow_range(delta)?;

        Ok(&mut self.inner_mut()[result])
    }

    fn inner(&self) -> &[u8] {
        // DO NOT convert this to impl Deref (internal use only)
        &*self.inner
    }

    fn inner_mut(&mut self) -> &mut [u8] {
        // DO NOT convert this to impl DerefMut (internal use only)
        &mut *self.inner
    }
}

impl<A: Array<Item = u8>> Buffer for ArrayVec<A> {
    fn resize_zero(&mut self, len: usize) {
        self.truncate(len);
        while self.len() < len {
            self.push(0);
        }
    }

    fn capacity_limit(&self) -> Option<usize> {
        Some(self.capacity())
    }
}

#[cfg(feature = "alloc")]
impl Buffer for Vec<u8> {
    fn resize_zero(&mut self, len: usize) {
        self.truncate(len);
        if self.len() < len {
            self.resize(len, 0);
        }
    }

    fn capacity_limit(&self) -> Option<usize> {
        None
    }
}

impl<A: Array<Item = u8>> TryFrom<ArrayVec<A>> for Sink<ArrayVec<A>> {
    type Error = SinkError;
    fn try_from(inner: ArrayVec<A>) -> Result<Self, Self::Error> {
        let limit = inner.capacity();
        Self::with_limit(
            inner,
            limit
                .try_into()
                .map_err(|_| SinkError::CapacityLimitTooHigh(limit))?,
        )
    }
}

#[cfg(feature = "alloc")]
impl TryFrom<Vec<u8>> for Sink<Vec<u8>> {
    type Error = SinkError;
    fn try_from(inner: Vec<u8>) -> Result<Self, Self::Error> {
        Self::new0(inner, None)
    }
}

/// A trait for builders.
pub trait Builder<B: Buffer>: Sized {
    fn sink(&mut self) -> &mut Sink<B>;
}

/// Starts a new builder with the given [`Builder`] as its parent.
pub trait PushBuilder<B: Buffer, P: Builder<B>>: Builder<B> {
    type Error;
    fn push(parent: P) -> Result<Self, Self::Error>;
}

/// Starts a new builder with the given [`Sink`] as its parent (or a [`Buffer`] for convenience).
pub trait NewBuilder<B: Buffer>: PushBuilder<B, Sink<B>> {
    fn new(sink: Sink<B>) -> Result<Self, Self::Error> {
        Ok(Self::push(sink)?)
    }
}

impl<B: Buffer, T: PushBuilder<B, Sink<B>>> NewBuilder<B> for T {}

impl<B: Buffer> Builder<B> for Sink<B> {
    fn sink(&mut self) -> &mut Sink<B> {
        self
    }
}

#[cfg(test)]
mod test {
    use core::convert::{TryFrom, TryInto};

    use arrayvec::ArrayVec;
    use assert_matches::assert_matches;

    use super::message::MessageBuilder;
    use super::name::NameBuilder;
    use super::{NewBuilder, Sink, SinkError};

    declare_any_error!(AnyError);

    fn a12() -> ArrayVec<[u8; 4096]> {
        ArrayVec::new()
    }

    fn a16() -> ArrayVec<[u8; 65536]> {
        ArrayVec::new()
    }

    #[test]
    fn empty() {
        assert_matches!(
            Sink::try_from(ArrayVec::from([0])).and(Ok(())),
            Err(SinkError::InnerNotEmpty(1))
        );
    }

    #[rustfmt::skip]
    #[test]
    fn limit() -> Result<(), AnyError> {
        use core::num::NonZeroU16;
        assert_eq!(Sink::try_from(a12())?.limit.map(NonZeroU16::get), Some(4096));
        assert_eq!(Sink::with_limit(a12(), 0)?.limit.map(NonZeroU16::get), Some(512));
        assert_eq!(Sink::with_limit(a12(), 1)?.limit.map(NonZeroU16::get), Some(512));
        assert_eq!(Sink::with_limit(a12(), 511)?.limit.map(NonZeroU16::get), Some(512));
        assert_eq!(Sink::with_limit(a12(), 512)?.limit.map(NonZeroU16::get), Some(512));
        assert_eq!(Sink::with_limit(a12(), 513)?.limit.map(NonZeroU16::get), Some(513));
        assert_eq!(Sink::with_limit(a12(), 4096)?.limit.map(NonZeroU16::get), Some(4096));
        assert_eq!(Sink::try_from(a12())?.limit.map(NonZeroU16::get), Some(4096));
        assert_matches!(Sink::with_limit(a12(), 4097).and(Ok(())), Err(SinkError::LimitTooHigh(Some(x), 4096)) if x == NonZeroU16::new(4097).unwrap());

        assert_eq!(Sink::with_limit(a16(), 65535)?.limit.map(NonZeroU16::get), Some(65535));
        assert_matches!(Sink::try_from(a16()).and(Ok(())), Err(SinkError::CapacityLimitTooHigh(65536)));

        Ok(())
    }

    #[test]
    #[rustfmt::skip]
    fn message() -> Result<(), AnyError> {
        assert_eq!(*MessageBuilder::new(a12().try_into()?)?.finish().finish(), [0; 12][..]);

        MessageBuilder::new(a12().try_into()?)?
            .question()?.qname()?.finish()?.finish("ANY".parse()?, "IN".parse()?)?
            .question()?.qname()?.finish()?.finish("ANY".parse()?, "IN".parse()?)?
            .into_an()
            .record()?.name()?.finish()?.try_into_data()?.finish()?
            .record()?.name()?.finish()?.try_into_data()?.finish()?
            .into_ns()
            .record()?.name()?.finish()?.try_into_data()?.finish()?
            .record()?.name()?.finish()?.try_into_data()?.finish()?
            .into_ar()
            .record()?.name()?.finish()?.try_into_data()?.finish()?
            .record()?.name()?.finish()?.try_into_data()?.finish()?
            .finish().finish();

        Ok(())
    }

    #[test]
    fn question() -> Result<(), AnyError> {
        #[rustfmt::skip]
        assert_eq!(*MessageBuilder::new(a12().try_into()?)?.question()?.qname()?.finish()?.finish("ANY".parse()?, "IN".parse()?)?.finish().finish(),
                   b"\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\0\x00\xFF\x00\x01"[..]);

        Ok(())
    }

    #[rustfmt::skip]
    #[test]
    fn name() -> Result<(), AnyError> {
        use super::name::NameError;

        assert_matches!(NameBuilder::new(a12().try_into()?)?.label(&[0; 0]).and(Ok(())),
                        Err(NameError::EmptyLabel));

        assert_matches!(NameBuilder::new(a12().try_into()?)?.label(&[0; 1]).and(Ok(())),
                        Ok(()));

        assert_matches!(NameBuilder::new(a12().try_into()?)?.label(&[0; 63]).and(Ok(())),
                        Ok(()));

        assert_matches!(NameBuilder::new(a12().try_into()?)?.label(&[0; 64]).and(Ok(())),
                        Err(NameError::LabelTooLong));

        NameBuilder::new(a12().try_into()?)?
            .label(&[0; 63])?.label(&[0; 63])?.label(&[0; 63])?.label(&[0; 62])?
            .finish()?.finish();

        assert_matches!(
            NameBuilder::new(a12().try_into()?)?
                .label(&[0; 63])?
                .label(&[0; 63])?
                .label(&[0; 63])?
                .label(&[0; 63])
                .and(Ok(())),
            Err(NameError::NameTooLong));

        assert_eq!(*NameBuilder::new(a12().try_into()?)?.label(b"daria")?.label(b"daz")?.label(b"cat")?.finish()?.finish(),
                   b"\x05daria\x03daz\x03cat\0"[..]);

        assert_eq!(*NameBuilder::new(a12().try_into()?)?.labels(br"")?.finish()?.finish(), b"\0"[..]);
        assert_eq!(*NameBuilder::new(a12().try_into()?)?.labels(br"daria.daz.cat.")?.finish()?.finish(), b"\x05daria\x03daz\x03cat\0"[..]);
        assert_eq!(*NameBuilder::new(a12().try_into()?)?.labels(br"Action\.domains.")?.finish()?.finish(), b"\x0EAction.domains\0"[..]);
        assert_eq!(*NameBuilder::new(a12().try_into()?)?.labels(br"\000.")?.finish()?.finish(), b"\x01\0\0"[..]);
        assert_eq!(*NameBuilder::new(a12().try_into()?)?.labels(br"\255.")?.finish()?.finish(), b"\x01\xFF\0"[..]);

        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br".").and(Ok(())), Err(NameError::EmptyLabel));
        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br"x").and(Ok(())), Err(NameError::UnfinishedLabel));
        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br"\").and(Ok(())), Err(NameError::UnfinishedEscape));
        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br"\0").and(Ok(())), Err(NameError::UnfinishedEscape));
        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br"\00").and(Ok(())), Err(NameError::UnfinishedEscape));
        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br"\00.").and(Ok(())), Err(NameError::DecimalEscapeSyntax(b'.')));
        assert_matches!(NameBuilder::new(a12().try_into()?)?.labels(br"\256").and(Ok(())), Err(NameError::DecimalEscapeRange(256)));

        Ok(())
    }
}

#[cfg(all(test, feature = "bench"))]
mod bench {
    mod build {
        extern crate test;

        use core::convert::TryInto;
        use test::Bencher;

        use super::super::message::MessageBuilder;
        use super::super::name::NameBuilder;
        use super::super::NewBuilder;
        use crate::core::Ttl;

        declare_any_error!(AnyError);

        #[bench]
        fn query(bencher: &mut Bencher) {
            bencher.iter(|| -> Result<usize, AnyError> {
                Ok(MessageBuilder::new(alloc::vec![].try_into()?)?
                    .id(0x1313)
                    .qr(false)
                    .rd(true)
                    .question()?
                    .qname()?
                    .labels(b"daria.daz.cat.")?
                    .finish()?
                    .finish("A".parse()?, "IN".parse()?)?
                    .into_ar()
                    .extension()?
                    .finish()?
                    .finish()
                    .finish()
                    .len())
            });
        }

        #[rustfmt::skip]
        #[bench]
        fn roots(bencher: &mut Bencher) {
            bencher.iter(|| -> Result<usize, AnyError> {
                Ok(MessageBuilder::new(alloc::vec![].try_into()?)?
                    .id(0x1313)
                    .qr(true)
                    .rd(true)
                    .ra(true)
                    .question()?.qname()?.finish()?.finish("NS".parse()?, "IN".parse()?)?
                    .into_an()
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"a.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"b.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"c.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"d.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"e.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"f.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"g.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"h.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"i.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"j.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"k.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"l.root-servers.net.")?.finish()?.finish()?
                    .record()?.name()?.finish()?.try_into_data()?.r#type("NS".parse()?).class("IN".parse()?).ttl(Ttl(518400)).build_rdata::<NameBuilder<_, _>>()?.labels(b"m.root-servers.net.")?.finish()?.finish()?
                    .into_ar()
                    .extension()?.finish()?
                    .finish()
                    .finish()
                    .len())
            });
        }

        #[bench]
        fn name(bencher: &mut Bencher) {
            bencher.iter(|| -> Result<usize, AnyError> {
                Ok(NameBuilder::new(alloc::vec![].try_into()?)?
                    .label(b"daria")?
                    .label(b"daz")?
                    .label(b"cat")?
                    .finish()?
                    .finish()
                    .len())
            });
        }
    }

    mod parse {
        extern crate test;

        use core::convert::TryInto;
        use test::Bencher;

        use super::super::name::NameBuilder;
        use super::super::NewBuilder;
        use crate::core::{Class, Type};

        declare_any_error!(AnyError);

        #[bench]
        fn name(bencher: &mut Bencher) {
            bencher.iter(|| -> Result<usize, AnyError> {
                Ok(NameBuilder::new(alloc::vec![].try_into()?)?
                    .labels(b"daria.daz.cat.")?
                    .finish()?
                    .finish()
                    .len())
            });
        }

        #[bench]
        fn r#type(bencher: &mut Bencher) {
            bencher.iter(|| "cname".parse::<Type>());
        }

        #[bench]
        fn class(bencher: &mut Bencher) {
            bencher.iter(|| "hs".parse::<Class>());
        }
    }
}
