use core::convert::Infallible;
use core::marker::PhantomData;
use core::ops::Range;

use byteorder::{ByteOrder, NetworkEndian};

use crate::emit::extension::{ExtensionBuilder, ExtensionError};
use crate::emit::name::NameError;
use crate::emit::question::{QuestionBuilder, QuestionError};
use crate::emit::record::{RecordBuilder, RecordError, RecordName};
use crate::emit::{Buffer, Builder, GrowError, PushBuilder, Sink};

error!(MessageError, Extension);
/// failed to emit message
#[derive(Debug, displaydoc::Display)]
#[prefix_enum_doc_attributes]
pub enum MessageError {
    /// FIXME rust-lang/rust#51085
    Infallible(Infallible),

    /// not enough space
    Grow(GrowError),

    /// too many questions in question section
    QdTooManyQuestions,

    /// too many records in answer section
    AnTooManyRecords,

    /// too many records in authority section
    NsTooManyRecords,

    /// too many records in additional section
    ArTooManyRecords,

    /// error while emitting OPT RR
    Extension(ExtensionError),

    /// error while emitting record
    Record(RecordError),

    /// error while emitting question
    Question(QuestionError),

    /// error while emitting record
    Name(NameError),
}

/// # Examples
///
/// ```rust
/// # #[macro_use] extern crate nonymous;
/// # declare_any_error!(AnyError);
/// use core::convert::TryInto;
/// use nonymous::emit::message::MessageBuilder;
/// use nonymous::emit::{NewBuilder, Sink};
/// let packet = MessageBuilder::new(arrayvec::ArrayVec::<[_; 4096]>::new().try_into()?)?
///     .id(0x1313).rd(true)
///     .question()?.qname()?.labels(b"cat.")?.finish()?.finish("NS".parse()?, "IN".parse()?)?
///     .into_ar()
///     .extension()?
///     .finish()?.finish().finish();
/// assert_eq!(packet.len(), 12 + 5 + 4 + 1 + 10);
/// # Ok::<(), AnyError>(())
/// ```
pub struct MessageBuilder<B: Buffer, P: Builder<B>, Q> {
    buffer: PhantomData<B>,
    parent: P,
    #[allow(dead_code)]
    section: Q,
    header: Range<usize>,
}

pub struct QdSection;
pub struct AnSection;
pub struct NsSection;
pub struct ArSection;
pub struct ArWithOpt;

pub trait QuestionSection {}
pub trait RecordSection {}

impl QuestionSection for QdSection {}
impl RecordSection for AnSection {}
impl RecordSection for NsSection {}
impl RecordSection for ArSection {}
impl RecordSection for ArWithOpt {}

impl<B: Buffer> PushBuilder<B, Sink<B>> for MessageBuilder<B, Sink<B>, QdSection> {
    type Error = MessageError;
    fn push(mut parent: Sink<B>) -> Result<Self, MessageError> {
        let header = parent.sink().grow_range(12).map_err(MessageError::Grow)?;

        Ok(Self {
            buffer: PhantomData,
            parent,
            section: QdSection,
            header,
        })
    }
}

builder! {
    <B, P> MessageBuilder {
        Builder [Q];
        @ <P> Q [Q]:
            /// Finish building the message.
            pub fn finish(mut self) -> P = {
                self.parent
            }

            pub fn id(mut self, value: u16) -> Self = {
                let offset = self.header.start;
                NetworkEndian::write_u16(&mut self.sink().inner_mut()[offset..], value);

                self
            }

            pub fn qr(mut self, value: bool) -> Self = {
                let offset = self.header.start + 2;

                self.set(offset, 7, value)
            }

            pub fn aa(mut self, value: bool) -> Self = {
                let offset = self.header.start + 2;

                self.set(offset, 2, value)
            }

            pub fn tc(mut self, value: bool) -> Self = {
                let offset = self.header.start + 2;

                self.set(offset, 1, value)
            }

            pub fn rd(mut self, value: bool) -> Self = {
                let offset = self.header.start + 2;

                self.set(offset, 0, value)
            }

            pub fn ra(mut self, value: bool) -> Self = {
                let offset = self.header.start + 3;

                self.set(offset, 7, value)
            }

            fn qd_increment(mut self) -> Result<Self, MessageError> = {
                let offset = self.header.start + 4;

                self.u16_increment(offset)
                    .or(Err(MessageError::QdTooManyQuestions))
            }

            fn an_increment(mut self) -> Result<Self, MessageError> = {
                let offset = self.header.start + 6;

                self.u16_increment(offset)
                    .or(Err(MessageError::AnTooManyRecords))
            }

            fn ns_increment(mut self) -> Result<Self, MessageError> = {
                let offset = self.header.start + 8;

                self.u16_increment(offset)
                    .or(Err(MessageError::NsTooManyRecords))
            }

            fn ar_increment(mut self) -> Result<Self, MessageError> = {
                let offset = self.header.start + 10;

                self.u16_increment(offset)
                    .or(Err(MessageError::ArTooManyRecords))
            }

            fn set(mut self, offset: usize, shift: usize, value: bool) -> Self = {
                self.sink().inner_mut()[offset] &= !(1 << shift);
                self.sink().inner_mut()[offset] |= (value as u8) << shift;

                self
            }

            fn u16_increment(mut self, offset: usize) -> Result<Self, ()> = {
                let x = NetworkEndian::read_u16(&self.sink().inner()[offset..]);
                let x = x.checked_add(1).ok_or(())?;
                NetworkEndian::write_u16(&mut self.sink().inner_mut()[offset..], x);

                Ok(self)
            }
        @ <P> QdSection:
            /// Starts a new [`RecordBuilder`](super::record::RecordBuilder) on the end of the message’s question section.
            pub fn question(mut self) = [push QuestionBuilder | MessageError::Question] { self.qd_increment()? }
            pub fn into_an(mut self) = [into AnSection] { self }
            pub fn into_ns(mut self) = [into NsSection] { self }
            pub fn into_ar(mut self) = [into ArSection] { self }
        @ <P> AnSection:
            /// Starts a new [`RecordBuilder`](super::record::RecordBuilder) on the end of the message’s answer section.
            pub fn record(mut self) = [push RecordBuilder<RecordName> | MessageError::Record] { self.an_increment()? }
            pub fn into_ns(mut self) = [into NsSection] { self }
            pub fn into_ar(mut self) = [into ArSection] { self }
        @ <P> NsSection:
            /// Starts a new [`RecordBuilder`](super::record::RecordBuilder) on the end of the message’s authority section.
            pub fn record(mut self) = [push RecordBuilder<RecordName> | MessageError::Record] { self.ns_increment()? }
            pub fn into_ar(mut self) = [into ArSection] { self }
        @ <P> ArSection:
            /// Starts a new [`RecordBuilder`](super::record::RecordBuilder) on the end of the message’s additional section.
            pub fn record(mut self) = [push RecordBuilder<RecordName> | MessageError::Record] { self.ar_increment()? }

            /// Start building a new EDNS OPT RR on the end of the message’s additional section.
            pub fn extension(mut self) = [push ExtensionBuilder | MessageError::Extension] {
                self.record()?
                    .name()
                    .map_err(MessageError::Record)?
                    .finish()
                    .map_err(MessageError::Name)?
                    .try_into_data()
                    .map_err(MessageError::Record)?
            }
        @ <P> ArWithOpt:
            /// Starts a new [`RecordBuilder`](super::record::RecordBuilder) on the end of the message’s additional section.
            pub fn record(mut self) = [push RecordBuilder<RecordName> | MessageError::Record] { self.ar_increment()? }
    }
}

transition! {
    MessageBuilder.section {
        (buffer, parent, header) QdSection -> AnSection;
        (buffer, parent, header) QdSection -> NsSection;
        (buffer, parent, header) QdSection -> ArSection;
        (buffer, parent, header) AnSection -> NsSection;
        (buffer, parent, header) AnSection -> ArSection;
        (buffer, parent, header) NsSection -> ArSection;
        (buffer, parent, header) ArSection -> ArWithOpt;
    }
}
