use core::str::FromStr;

use crate::r#const::{CLASS, TYPE};

error!(TypeFromStrError);
#[derive(Debug, displaydoc::Display)]
/// unrecognised TYPE name
pub struct TypeFromStrError;

error!(ClassFromStrError);
#[derive(Debug, displaydoc::Display)]
/// unrecognised CLASS name
pub struct ClassFromStrError;

error!(TtlFromStrError);
#[derive(Debug, displaydoc::Display)]
/// failed to parse TTL value
pub struct TtlFromStrError;

#[derive(Debug, Clone)]
pub struct Type(pub u16);

#[derive(Debug, Clone)]
pub struct Class(pub u16);

#[derive(Debug, Clone)]
pub struct Ttl(pub u32);

impl Type {
    pub fn value(&self) -> u16 {
        self.0
    }
}

impl Class {
    pub fn value(&self) -> u16 {
        self.0
    }
}

impl Ttl {
    pub fn value(&self) -> u32 {
        self.0
    }
}

impl FromStr for Type {
    type Err = TypeFromStrError;

    fn from_str(source: &str) -> Result<Self, Self::Err> {
        if let Some(value) = TYPE.number(&source) {
            return Ok(Type(value));
        }

        // RFC 3597 § 5
        if source.starts_with("TYPE") {
            return Ok(Type(
                u16::from_str_radix(&source[4..], 10).map_err(|_| TypeFromStrError)?,
            ));
        }

        Err(TypeFromStrError)
    }
}

impl FromStr for Class {
    type Err = ClassFromStrError;

    fn from_str(source: &str) -> Result<Self, Self::Err> {
        if let Some(value) = CLASS.number(&source) {
            return Ok(Class(value));
        }

        // RFC 3597 § 5
        if source.starts_with("CLASS") {
            return Ok(Class(
                u16::from_str_radix(&source[5..], 10).map_err(|_| ClassFromStrError)?,
            ));
        }

        Err(ClassFromStrError)
    }
}

#[cfg(all(test, feature = "bench"))]
mod bench {
    mod parse {
        extern crate test;

        use test::Bencher;

        use crate::core::{Class, Type};

        #[bench]
        fn r#type(bencher: &mut Bencher) {
            bencher.iter(|| "cname".parse::<Type>());
        }

        #[bench]
        fn class(bencher: &mut Bencher) {
            bencher.iter(|| "hs".parse::<Class>());
        }
    }
}
