use core::marker::PhantomData;

use byteorder::{ByteOrder, NetworkEndian};

use crate::core::{Class, Type};
use crate::emit::message::{MessageBuilder, QuestionSection};
use crate::emit::name::{NameBuilder, NameError};
use crate::emit::{Buffer, Builder, GrowError, PushBuilder, Sink};

error!(QuestionError);
/// failed to emit question
#[derive(Debug, displaydoc::Display)]
#[prefix_enum_doc_attributes]
pub enum QuestionError {
    /// not enough space
    Grow(GrowError),

    /// error while emitting name
    Name(NameError),
}

pub struct QuestionBuilder<B: Buffer, P: Builder<B>> {
    buffer: PhantomData<B>,
    parent: P,
}

impl<B: Buffer, P: Builder<B>> PushBuilder<B, P> for QuestionBuilder<B, P> {
    type Error = QuestionError;
    fn push(parent: P) -> Result<Self, QuestionError> {
        Ok(Self {
            buffer: PhantomData,
            parent,
        })
    }
}

builder! {
    <B, P> QuestionBuilder {
        Builder;
        @ <P>:
            /// Start building the record’s QNAME.
            pub fn qname(mut self) = [push NameBuilder | QuestionError::Name] { self }

        @ <MessageBuilder<B, P, Q>> [Q: QuestionSection]:
            /// Finish building the question and return to the message’s question section.
            pub fn finish(
                mut self,
                qtype: Type,
                qclass: Class,
            ) -> Result<MessageBuilder<B, P, Q>, QuestionError> = {
                NetworkEndian::write_u16(
                    self.sink().grow_mut(2).map_err(QuestionError::Grow)?,
                    qtype.value(),
                );
                NetworkEndian::write_u16(
                    self.sink().grow_mut(2).map_err(QuestionError::Grow)?,
                    qclass.value(),
                );

                Ok(self.parent)
            }
    }
}
