/*
    - Author: Molese <molese@protonmail.com> 2022
    - License: GPL 2.0
*/

//! Personal Standard Rust Lib Publics for reuse and to avoid redefinition

#![deny(
    missing_debug_implementations,
    missing_copy_implementations,
    trivial_casts,
    trivial_numeric_casts,
    unstable_features,
    unused_import_braces,
    unused_qualifications
)]
#![deny(clippy::all)]

/* simplified assertion */
/// Exits with error message if condition is true
#[macro_export]
macro_rules! die {
    ($exit:expr, $exit_code:expr, $($arg:tt)*) => {
        {
            use std::io::Write;
            let stderr = &mut std::io::stderr();
            let errmsg = format!($($arg)*);
            writeln!(stderr, "Fatal: {}!", errmsg).unwrap();
            if $exit {
                std::process::exit($exit_code);
            }
        }
    };
}

/// Returns product of all elements in a list
#[macro_export]
macro_rules! product {
    ($x:expr) => {
        $x.iter().fold(1, |acc, x| acc * x)
    };
}

/// Returns a random boolean value.
pub fn rand_bool() -> bool {
    use fastrand::bool as fastbool;
    fastbool()
}

/// Taking n as u128, returns true or false in case of n being even or odd respectively.
pub fn is_even(n: u128) -> bool {
    n % 2 == 0
}

/// Returns true if n is even or false if n is odd.
pub fn is_even_any(n: &dyn ToString) -> bool {
    is_even(n.to_string().parse::<u128>().unwrap())
}

/// Converts any type to String.
pub fn any_to_string<T: ToString>(s: T) -> String {
    s.to_string()
}

/// Return true if cmd is executable else false.
pub fn is_executable(cmd: &str) -> bool {
    use std::process::Command;
    Command::new(cmd).output().is_ok()
}

/// Raises base to the power of exp and returns the result.
pub fn power(base: u128, exp: u128) -> u128 {
    base.pow(exp.try_into().unwrap())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn product_vec() {
        assert_eq!(product!(vec![1, 2, 3, 4]), 24);
    }

    #[test]
    fn product_slice() {
        assert_eq!(product!([1, 2, 3, 4]), 24);
    }

    #[test]
    fn rand_bool_obvious() {
        let rand_bool: bool = rand_bool();
        assert_eq!(rand_bool, rand_bool);
    }

    #[test]
    fn evens() {
        for i in (0..=100).step_by(2) {
            assert!(is_even(i));
        }
    }

    #[test]
    fn odds() {
        for i in (1..=20).step_by(2) {
            assert!(!is_even(i));
        }
    }

    #[test]
    fn is_even_any_100() {
        for n in 0..100 {
            if is_even(n) {
                assert!(is_even_any(&n));
            } else {
                assert!(!is_even_any(&n));
            }
        }
    }

    #[test]
    fn any_to_string_all() {
        assert_eq!(any_to_string(0), "0".to_string());
        assert_eq!(any_to_string("0".to_string()), "0".to_string());
        assert_eq!(any_to_string(0u8), "0".to_string());
        assert_eq!(any_to_string(0u16), "0".to_string());
        assert_eq!(any_to_string(0u32), "0".to_string());
        assert_eq!(any_to_string(0u64), "0".to_string());
        assert_eq!(any_to_string(0u128), "0".to_string());
        assert_eq!(any_to_string(0usize), "0".to_string());
        assert_eq!(any_to_string(0i8), "0".to_string());
        assert_eq!(any_to_string(0i16), "0".to_string());
        assert_eq!(any_to_string(0i32), "0".to_string());
        assert_eq!(any_to_string(0i64), "0".to_string());
        assert_eq!(any_to_string(0i128), "0".to_string());
        assert_eq!(any_to_string(0isize), "0".to_string());
        assert_eq!(any_to_string(0.0f32), "0".to_string());
        assert_eq!(any_to_string(0.0f64), "0".to_string());
    }

    #[test]
    fn is_executable_cmds() {
        assert!(is_executable("ls"));
        assert!(!is_executable("lsd"));
    }

    #[test]
    fn power_mem() {
        assert_eq!(power(2, 3), 8);
        assert_eq!(power(2, 4), 16);
        assert_eq!(power(2, 5), 32);
        assert_eq!(power(2, 6), 64);
        assert_eq!(power(2, 7), 128);
        assert_eq!(power(2, 8), 256);
        assert_eq!(power(2, 9), 512);
        assert_eq!(power(2, 10), 1024);
        assert_eq!(power(2, 11), 2048);
    }
}
