// Copyright (C) 2022 Andreas Doerr
//
// SPDX-License-Identifier: Apache-2.0

use crate::Needed;

/// Take bytes out of an input slice
pub trait Take: Sized {
    /// Return a slice of `count` bytes
    ///
    /// ## Panics
    ///
    /// Panics if count > len
    fn take(&self, count: usize) -> Self;

    /// Split a slice at `count` index
    ///
    /// Return a tuple with two halves. The tuple will contain the following
    /// elements `([count, len), [0, count))`
    ///
    /// ## Panics
    ///
    /// Panics if count > len
    fn take_split(&self, count: usize) -> (Self, Self);
}

impl Take for &[u8] {
    fn take(&self, count: usize) -> Self {
        &self[0..count]
    }

    fn take_split(&self, count: usize) -> (Self, Self) {
        let (prefix, sufix) = self.split_at(count);

        (sufix, prefix)
    }
}

/// Comparision result type
#[derive(Debug, PartialEq)]
pub enum Comparision {
    /// Operands are equal
    Equal,
    /// Operands are unequal
    Unequal,
    /// Operands differ in number of elements
    Incomplete,
}

/// Compare operation trait
///
/// The reason why we don't just use [`std::cmp::PartialEq`] is because we want
/// to know whether `self` and `T` are unequal or just differ in the number of
/// elements.
pub trait Compare<T> {
    fn compare(&self, other: T) -> Comparision;
}

impl Compare<&[u8]> for &[u8] {
    fn compare(&self, other: &[u8]) -> Comparision {
        let pos = self.iter().zip(other.iter()).position(|(a, b)| a != b);

        match pos {
            None if self.len() >= other.len() => Comparision::Equal,
            None => Comparision::Incomplete,
            Some(_) => Comparision::Unequal,
        }
    }
}

/// Number of elements operation trait
pub trait Length {
    fn length(&self) -> usize;
}

impl Length for &[u8] {
    fn length(&self) -> usize {
        self.len()
    }
}

pub trait Index {
    type Item;

    fn index(&self, idx: usize) -> std::result::Result<usize, Needed>;
}

impl Index for &[u8] {
    type Item = u8;

    fn index(&self, idx: usize) -> std::result::Result<usize, Needed> {
        if self.len() >= idx {
            Ok(idx)
        } else {
            Err(Needed::Size(idx - self.len()))
        }
    }
}
