// Copyright (C) 2022 Andreas Doerr
//
// SPDX-License-Identifier: Apache-2.0

#[derive(Debug)]
pub enum Needed {
    Unknown,
    Size(usize),
}

/// The `Terminate` enum indicates that the parser was not successful
#[derive(Debug)]
pub enum Terminate<T> {
    /// Not enough data
    Incomplete(Needed),
    /// Recoverable error
    Error(T),
    /// Unrecoverable error, need to backtrack
    Failure(T),
}

impl<E> std::fmt::Display for Terminate<E>
where
    E: std::fmt::Display,
{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Terminate::Incomplete(Needed::Size(u)) => write!(f, "requires {} bytes/chars", u),
            Terminate::Incomplete(Needed::Unknown) => write!(f, "requires more data"),
            Terminate::Error(e) => write!(f, "error: {}", e),
            Terminate::Failure(e) => write!(f, "failure: {}", e),
        }
    }
}

/// Default error type to be used with [`Terminate::Error`] and [`Terminate::Failure`]
#[derive(Debug, thiserror::Error)]
pub enum Error<I> {
    #[error("tag mismatch at `{0}`")]
    Tag(I),
}

/// Result type for parsing combinator functions
pub type Result<I, O, E = Error<I>> = std::result::Result<(I, O), Terminate<E>>;
