use crate::*;

#[near_bindgen]
impl Contract {
  #[payable]
  pub fn nft_mint (
    &mut self,
    token_id: Option<TokenId>,
    metadata: TokenMetadata,
    perpetual_royalies: Option<HashMap<AccountId, u32>>,
    receiver_id: Option<ValidAccountId>,
    token_type: Option<TokenType>,
  ) {
    let mut final_token_id = format!("{}", self.token_metadata_by_id.len() + 1);
    if let Some(token_id) = token_id {
      final_token_id = token_id
    }

    let initial_storage_usage = env::storage_usage();
    let mut owner_id = env::predecessor_account_id();
    if let Some(receiver_id) = receiver_id {
      owner_id = receiver_id.into();
    }

    // CUSTOM - create royalty map
    let mut royalty = HashMap::new();
    let mut total_perpetual = 0;

    // user added perpetual_royalties (percentage paid with every transfer)
    if let Some(perpetual_royalties) = perpetual_royalties {
      assert!(perpetual_royalies.len() < 7, "Cannot add more than 6 perpetual royalty amounts");
      for (account, amount) in perpetual_royalties {
        royalty.insert(account, amount);
        total_perpetual += amount;
      }
    }

    if token_type.is_some() {
      let token_type = token_type.clone().unwrap();
      let cap = u64::from(*self.supply_cap_by_type.get(&token_type).expect("Token type must have supply cap."));
      let supply = u64::from(self.nft_supply_for_type(&token_type));
      assert!(supply < cap, "Cannot mint anymore of token type");
      let mut token_per_type = self
        .token_per_type
        .get(&token_type)
        .unwrap_or_else(|| {
          UnorderedSet::new(
            StorageKey::TokensPerTypeInner {
              token_type_hash: hash_account_id(&token_type),
            }
            .try_to_vec()
            .unwrap(),
          )
        });
        tokens_per_type.insert(&final_token_id);
        self.tokens_per_type.insert(&token_type, &tokens_per_type);
    }
    // END CUSTOM

    let token = Token {
      owner_id,
      approved_account_ids: Default::default(),
      next_approval_id: 0,
      royalty,
      token_type,
    };
    assert!(
      self.tokens_by_id.insert(&final_token_id, &token).is_none(),
      "Token already exists"
    );
    self.token_metadata_by_id.insert(&final_token_id, &metadata);
    self.internal_add_token_to_owner(&token.owner_id, &final_token_id);

    let new_token_size_in_bytes = env::storage_usage() - initial_storage_usage;
    let required_storage_in_bytes = self.extra_storage_in_bytes_per_token + new_token_size_in_bytes;

    refund_deposit(required_storage_in_bytes);
  }
}