# nfc1-sys
[![Crates.io](https://img.shields.io/crates/v/nfc1-sys)](https://crates.io/crates/nfc1-sys)

This crate provides low-level bindings to [`libnfc`](https://github.com/alexrsagen/libnfc), generated by [`bindgen`](https://crates.io/crates/bindgen).

In contrast to [`nfc-sys`](https://github.com/dsgriffin/nfc-sys), this crate additionally provides:
- [Metadata](https://doc.rust-lang.org/cargo/reference/build-scripts.html#the-links-manifest-key) which allows dependent crates to find the `nfc/nfc.h` header, compile native code that depends on `libnfc` or link to it in Rust code.
- Vendored submodule copy of `libnfc` (with build tweaks for `x86_64-pc-windows-msvc`), which means you don't have to separately install `libnfc` to use this crate.

## Usage
Add `nfc1-sys` as a dependency in your project's `Cargo.toml` file:
```toml
[dependencies]
nfc1-sys = "0.1"
```

Import the `nfc1_sys` crate in your project, then you can use all functions starting with `nfc_` from `libnfc`.

See the [`libnfc` wiki](https://github.com/nfc-tools/libnfc/wiki) or [`libnfc` 1.8.0 examples](https://github.com/nfc-tools/libnfc/tree/libnfc-1.8.0/examples) for information on how to use it. The API is the same, as this is just a binding.

### Usage example
Note that there is quite a lot of unsafe code here. This is because this is just a simple binding, not a safe wrapper.

```rust
extern crate nfc1_sys;
use std::ffi::CStr;
use std::mem::MaybeUninit;
use nfc1_sys::{nfc_context, nfc_device, nfc_version, nfc_init, nfc_open, nfc_device_get_name, nfc_close, nfc_exit};

fn main() {
	let version = unsafe { CStr::from_ptr(nfc_version()) }.to_str().unwrap();
	println!("libnfc v{}", version);

	let context: *mut nfc_context = unsafe {
		let mut ctx = MaybeUninit::uninit();
		nfc_init(ctx.as_mut_ptr());
		ctx.assume_init()
	};

	let device: *mut nfc_device = unsafe {
		let pnd = nfc_open(context, std::ptr::null_mut());
		if pnd.as_ref().is_none() {
			nfc_exit(context);
			panic!("Error opening NFC reader");
		}
		pnd
	};

	let device_name = unsafe { CStr::from_ptr(nfc_device_get_name(device)) }.to_str().unwrap();
	println!("NFC reader: {} opened", device_name);

	unsafe {
		nfc_close(device);
		nfc_exit(context);
	}
}
```
