use borsh::{BorshDeserialize, BorshSerialize};
use solana_program::{
    instruction::{AccountMeta, Instruction},
    pubkey::Pubkey,
    sysvar::rent,
};

#[repr(C)]
#[derive(BorshSerialize, BorshDeserialize, PartialEq, Debug, Clone)]
pub struct Winner {
    pub address: Pubkey,
    pub claimed: bool,
}

#[repr(C)]
#[derive(BorshSerialize, BorshDeserialize, Clone, Debug, PartialEq)]
pub enum LotteryInstruction {
    /// 0. `[singer]` admin
    /// 1. `[writable]` vault
    /// 2. `[writable]` usdc token account of vault
    /// 3. `[]` rent Sysvar
    /// 5. `[]` token program
    InitVault,
    /// 0. `[signer]` owner of the stake
    /// 1. `[writable]` stake account
    /// 2. `[]` system program
    /// 3. `[]` rent Sysvar
    CreateStake,
    /// 0. `[signer]` owner of the stake
    /// 1. `[writable]` stake account
    /// 2. `[]` usdc mint
    /// 3. `[writable]` destination usdc token account of vault
    /// 4. `[writable]` source usdc token account of owner
    /// 5. `[]` token program
    DepositStake { amount: u64 },
    /// 0. `[signer]` owner of the stake
    /// 1. `[writable]` stake account
    /// 2. `[]` usdc mint
    /// 3. `[writable]` source usdc token account of vault
    /// 4. `[writable]` destination usdc token account of owner
    /// 5. `[]` vault
    /// 6. `[]` token program
    WithdrawStake { amount: u64 },
    // TODO add clock to check end_at
    /// 0. `[singer]` admin
    /// 1. `[writable]` epoch account
    /// 2. `[writable]` latest epoch account
    /// 3. `[]` system program
    /// 4. `[]` rent Sysvar
    CreateEpoch { index: u64, end_at: i64 },
    /// 0. `[singer]` admin
    /// 1. `[writable]` latest epoch account
    /// 2. `[]` system program
    /// 3. `[]` rent Sysvar
    InitLatestEpoch,
    /// 0. `[singer]` admin
    /// 1. `[writable]` lottery ticket account
    /// 2. `[]` epoch account
    /// 3. `[]` latest epoch account
    /// 4. `[]` owner account
    /// 5. `[]` system program
    /// 6. `[]` rent Sysvar
    CreateLotteryTicket { uri: String },
    /// 0. `[singer]` admin
    /// 1. `[writable]` lottery ticket account
    /// 2. `[]` epoch account
    /// 3. `[]` latest epoch account
    /// 4. `[]` system program
    /// 5. `[]` rent Sysvar
    DeclearEpochWinningCombination,
    /// 0. `[singer]` admin
    /// 1. `[writable]` epoch tier winners account
    /// 2. `[]` epoch account
    /// 3. `[]` latest epoch account
    /// 4. `[]` system program
    /// 5. `[]` rent Sysvar
    PublishEpochTierWinners {
        tier: u8,
        winning_size: u64,
        winners: Vec<Winner>,
    },

    /// 0. `[singer]` owner
    /// 1. `[writable]` epoch tier winners account
    /// 2. `[]` usdc mint
    /// 3. `[writable]` source usdc token account of vault
    /// 4. `[writable]` destination usdc token account of owner
    /// 5. `[]` vault
    /// 6. `[]` token program
    ClaimWinning,
}

pub fn init_vault(
    program_id: &Pubkey,
    admin: &Pubkey,
    vault: &Pubkey,
    vault_usdc_token: &Pubkey,
    usdc_mint: &Pubkey,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::InitVault,
        vec![
            AccountMeta::new(admin.clone(), true),
            AccountMeta::new(vault.clone(), false),
            AccountMeta::new(vault_usdc_token.clone(), false),
            AccountMeta::new(usdc_mint.clone(), false),
            AccountMeta::new_readonly(solana_program::system_program::id(), false),
            AccountMeta::new_readonly(rent::id(), false),
            AccountMeta::new_readonly(spl_token::id(), false),
        ],
    )
}

pub fn create_stake(program_id: &Pubkey, owner: &Pubkey, stake: &Pubkey) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::CreateStake,
        vec![
            AccountMeta::new(owner.clone(), true),
            AccountMeta::new(stake.clone(), false),
            AccountMeta::new_readonly(solana_program::system_program::id(), false),
            AccountMeta::new_readonly(rent::id(), false),
        ],
    )
}

pub fn deposit_stake(
    program_id: &Pubkey,
    owner: &Pubkey,
    stake: &Pubkey,
    usdc_mint: &Pubkey,
    vault_usdc_token: &Pubkey,
    owner_usdc_token: &Pubkey,
    amount: u64,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::DepositStake { amount },
        vec![
            AccountMeta::new(owner.clone(), true),
            AccountMeta::new(stake.clone(), false),
            AccountMeta::new(usdc_mint.clone(), false),
            AccountMeta::new(vault_usdc_token.clone(), false),
            AccountMeta::new(owner_usdc_token.clone(), false),
            AccountMeta::new_readonly(spl_token::id(), false),
        ],
    )
}

pub fn withdraw_stake(
    program_id: &Pubkey,
    owner: &Pubkey,
    stake: &Pubkey,
    usdc_mint: &Pubkey,
    vault_usdc_token: &Pubkey,
    owner_usdc_token: &Pubkey,
    vault: &Pubkey,
    amount: u64,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::WithdrawStake { amount },
        vec![
            AccountMeta::new(owner.clone(), true),
            AccountMeta::new(stake.clone(), false),
            AccountMeta::new_readonly(usdc_mint.clone(), false),
            AccountMeta::new(vault_usdc_token.clone(), false),
            AccountMeta::new(owner_usdc_token.clone(), false),
            AccountMeta::new_readonly(vault.clone(), false),
            AccountMeta::new_readonly(spl_token::id(), false),
        ],
    )
}

pub fn create_epoch(
    program_id: &Pubkey,
    admin: &Pubkey,
    epoch: &Pubkey,
    latest_epoch: &Pubkey,
    index: u64,
    end_at: i64,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::CreateEpoch { index, end_at },
        vec![
            AccountMeta::new(admin.clone(), true),
            AccountMeta::new(epoch.clone(), false),
            AccountMeta::new(latest_epoch.clone(), false),
            AccountMeta::new_readonly(solana_program::system_program::id(), false),
            AccountMeta::new_readonly(rent::id(), false),
        ],
    )
}

pub fn init_latest_epoch(
    program_id: &Pubkey,
    admin: &Pubkey,
    latest_epoch: &Pubkey,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::InitLatestEpoch,
        vec![
            AccountMeta::new(admin.clone(), true),
            AccountMeta::new(latest_epoch.clone(), false),
            AccountMeta::new_readonly(solana_program::system_program::id(), false),
            AccountMeta::new_readonly(rent::id(), false),
        ],
    )
}

pub fn create_lottery_ticket(
    program_id: &Pubkey,
    admin: &Pubkey,
    lottery_ticket: &Pubkey,
    epoch: &Pubkey,
    latest_epoch: &Pubkey,
    owner: &Pubkey,
    uri: String,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::CreateLotteryTicket { uri },
        vec![
            AccountMeta::new(admin.clone(), true),
            AccountMeta::new(lottery_ticket.clone(), false),
            AccountMeta::new_readonly(epoch.clone(), false),
            AccountMeta::new_readonly(latest_epoch.clone(), false),
            AccountMeta::new_readonly(owner.clone(), false),
            AccountMeta::new_readonly(solana_program::system_program::id(), false),
            AccountMeta::new_readonly(rent::id(), false),
        ],
    )
}

pub fn publish_epoch_tier_winners(
    program_id: &Pubkey,
    admin: &Pubkey,
    epoch_tier_winners: &Pubkey,
    epoch: &Pubkey,
    latest_epoch: &Pubkey,
    tier: u8,
    winning_size: u64,
    winners: Vec<Winner>,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::PublishEpochTierWinners {
            tier,
            winning_size,
            winners,
        },
        vec![
            AccountMeta::new(admin.clone(), true),
            AccountMeta::new(epoch_tier_winners.clone(), false),
            AccountMeta::new_readonly(epoch.clone(), false),
            AccountMeta::new_readonly(latest_epoch.clone(), false),
            AccountMeta::new_readonly(solana_program::system_program::id(), false),
            AccountMeta::new_readonly(rent::id(), false),
        ],
    )
}

pub fn claim_winning(
    program_id: &Pubkey,
    owner: &Pubkey,
    epoch_tier_winners: &Pubkey,
    usd_mint: &Pubkey,
    vault_usdc_token: &Pubkey,
    owner_usdc_token: &Pubkey,
    vault: &Pubkey,
) -> Instruction {
    Instruction::new_with_borsh(
        program_id.clone(),
        &LotteryInstruction::ClaimWinning,
        vec![
            AccountMeta::new(owner.clone(), true),
            AccountMeta::new(epoch_tier_winners.clone(), false),
            AccountMeta::new_readonly(usd_mint.clone(), false),
            AccountMeta::new(vault_usdc_token.clone(), false),
            AccountMeta::new(owner_usdc_token.clone(), false),
            AccountMeta::new_readonly(vault.clone(), false),
            AccountMeta::new_readonly(spl_token::id(), false),
        ],
    )
}
