// Copyright (c) The nextest Contributors
// SPDX-License-Identifier: MIT OR Apache-2.0

use camino::{Utf8Path, Utf8PathBuf};
use serde::{Deserialize, Serialize};
use std::{collections::BTreeMap, fmt, path::PathBuf, process::Command};

use crate::CommandError;

/// Command builder for `cargo nextest list`.
#[derive(Clone, Debug, Default)]
pub struct ListCommand {
    cargo_path: Option<Box<Utf8Path>>,
    manifest_path: Option<Box<Utf8Path>>,
    current_dir: Option<Box<Utf8Path>>,
    args: Vec<Box<str>>,
}

impl ListCommand {
    /// Creates a new `ListCommand`.
    ///
    /// This command runs `cargo nextest list`.
    pub fn new() -> Self {
        Self::default()
    }

    /// Path to `cargo` executable. If not set, this will use the the `$CARGO` environment variable, and
    /// if that is not set, will simply be `cargo`.
    pub fn cargo_path(&mut self, path: impl Into<Utf8PathBuf>) -> &mut Self {
        self.cargo_path = Some(path.into().into());
        self
    }

    /// Path to `Cargo.toml`.
    pub fn manifest_path(&mut self, path: impl Into<Utf8PathBuf>) -> &mut Self {
        self.manifest_path = Some(path.into().into());
        self
    }

    /// Current directory of the `cargo nextest list` process.
    pub fn current_dir(&mut self, path: impl Into<Utf8PathBuf>) -> &mut Self {
        self.current_dir = Some(path.into().into());
        self
    }

    /// Adds an argument to the end of `cargo nextest list`.
    pub fn add_arg(&mut self, arg: impl Into<String>) -> &mut Self {
        self.args.push(arg.into().into());
        self
    }

    /// Adds several arguments to the end of `cargo nextest list`.
    pub fn add_args(&mut self, args: impl IntoIterator<Item = impl Into<String>>) -> &mut Self {
        for arg in args {
            self.add_arg(arg.into());
        }
        self
    }

    /// Builds a command for `cargo nextest list`. This is the first part of the work of [`self.exec`].
    pub fn cargo_command(&self) -> Command {
        let cargo_path: PathBuf = self.cargo_path.as_ref().map_or_else(
            || std::env::var_os("CARGO").map_or("cargo".into(), PathBuf::from),
            |path| PathBuf::from(path.as_std_path()),
        );

        let mut command = Command::new(&cargo_path);
        if let Some(path) = &self.manifest_path.as_deref() {
            command.args(["--manifest-path", path.as_str()]);
        }
        if let Some(current_dir) = &self.current_dir.as_deref() {
            command.current_dir(current_dir);
        }

        command.args(["nextest", "list", "--format=json"]);

        command.args(self.args.iter().map(|s| s.as_ref()));
        command
    }

    /// Executes `cargo nextest list` and parses the output into a [`TestListSummary`].
    ///
    ///
    pub fn exec(&self) -> Result<TestListSummary, CommandError> {
        let mut command = self.cargo_command();
        let output = command.output().map_err(CommandError::Exec)?;

        if !output.status.success() {
            // The process exited with a non-zero code.
            let exit_code = output.status.code();
            let stderr = output.stderr;
            return Err(CommandError::CommandFailed { exit_code, stderr });
        }

        // Try parsing stdout.
        serde_json::from_slice(&output.stdout).map_err(CommandError::Json)
    }
}

/// Root element for a serializable list of tests generated by nextest.
#[derive(Clone, Debug, Default, Eq, PartialEq, Deserialize, Serialize)]
#[serde(rename_all = "kebab-case")]
#[non_exhaustive]
pub struct TestListSummary {
    /// Number of tests (including skipped and ignored) across all binaries.
    pub test_count: usize,

    /// A map of Rust test suites to the test binaries within them, keyed by a unique identifier
    /// for each test suite.
    pub rust_suites: BTreeMap<String, RustTestSuiteSummary>,
}

impl TestListSummary {
    /// Parse JSON output from `cargo nextest list --format json`.
    pub fn parse_json(json: impl AsRef<str>) -> Result<Self, serde_json::Error> {
        serde_json::from_str(json.as_ref())
    }
}

/// A serializable suite of tests within a Rust test binary.
///
/// Part of a [`TestListSummary`].
#[derive(Clone, Debug, Eq, PartialEq, Deserialize, Serialize)]
#[serde(rename_all = "kebab-case")]
pub struct RustTestSuiteSummary {
    /// The name of this package in the workspace.
    pub package_name: String,

    /// The name of the test binary within the package.
    pub binary_name: String,

    /// The unique package ID assigned by Cargo to this test.
    ///
    /// This package ID can be used for lookups in `cargo metadata`.
    pub package_id: String,

    /// The path to the test binary executable.
    pub binary_path: Utf8PathBuf,

    /// The working directory that tests within this package are run in.
    pub cwd: Utf8PathBuf,

    /// Test case names and other information about them.
    pub testcases: BTreeMap<String, RustTestCaseSummary>,
}

/// Serializable information about an individual test case within a Rust test suite.
///
/// Part of a [`RustTestSuiteSummary`].
#[derive(Clone, Debug, Eq, PartialEq, Deserialize, Serialize)]
#[serde(rename_all = "kebab-case")]
pub struct RustTestCaseSummary {
    /// Returns true if this test is marked ignored.
    ///
    /// Ignored tests, if run, are executed with the `--ignored` argument.
    pub ignored: bool,

    /// Whether the test matches the provided test filter.
    ///
    /// Only tests that match the filter are run.
    pub filter_match: FilterMatch,
}

/// An enum describing whether a test matches a filter.
#[derive(Copy, Clone, Debug, Eq, PartialEq, Deserialize, Serialize)]
#[serde(rename_all = "kebab-case", tag = "status")]
pub enum FilterMatch {
    /// This test matches this filter.
    Matches,

    /// This test does not match this filter.
    Mismatch {
        /// Describes the reason this filter isn't matched.
        reason: MismatchReason,
    },
}

impl FilterMatch {
    /// Returns true if the filter doesn't match.
    pub fn is_match(&self) -> bool {
        matches!(self, FilterMatch::Matches)
    }
}

/// The reason for why a test doesn't match a filter.
#[derive(Copy, Clone, Debug, Eq, PartialEq, Deserialize, Serialize)]
#[serde(rename_all = "kebab-case")]
#[non_exhaustive]
pub enum MismatchReason {
    /// This test does not match the run-ignored option in the filter.
    Ignored,

    /// This test does not match the provided string filters.
    String,

    /// This test is in a different partition.
    Partition,
}

impl fmt::Display for MismatchReason {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            MismatchReason::Ignored => write!(f, "does not match the run-ignored option"),
            MismatchReason::String => write!(f, "does not match the provided string filters"),
            MismatchReason::Partition => write!(f, "is in a different partition"),
        }
    }
}
