/// A mapping of known file extensions and their MIME types.
///
/// Required to be sorted lexicographically by extension for ease of maintenance.
///
/// Multiple MIME types per extension are supported; the order is arbitrary but the first should be
/// the most prevalent by most recent RFC declaration or explicit succession of other media types.
///
/// NOTE: when adding or modifying entries, please include a citation in the commit message.
/// If a media type for an extension changed by official IETF RFC, please keep the old entry but add
/// the new one before it in the slice literal, e.g.:
///
/// ```ignore
/// - ("md", &["text/x-markdown"]),
/// + ("md", &["text/markdown", "text/x-markdown"]),
/// ```
///
/// Sourced from:
/// https://github.com/samuelneff/MimeTypeMap/blob/master/MimeTypeMap.cs
/// https://github.com/jshttp/mime-db extracted with https://gist.github.com/soyuka/b7e29d359b2c14c21bdead923c01cc81

#[allow(dead_code)]
pub static MIME_TYPES: &[(&str, &[&str])] = &[
	("123", &["application/vnd.lotus-1-2-3"]),
	("1km", &["application/vnd.1000minds.decision-model+xml"]),
	("323", &["text/h323"]),
	("3dml", &["text/vnd.in3d.3dml"]),
	("3ds", &["image/x-3ds"]),
	("3g2", &["video/3gpp2"]),
	("3gp", &["video/3gpp"]),
	("3gp2", &["video/3gpp2"]),
	("3gpp", &["video/3gpp"]),
	("3mf", &["model/3mf"]),
	("7z", &["application/x-7z-compressed"]),
	("aa", &["audio/audible"]),
	("aab", &["application/x-authorware-bin"]),
	("aac", &["audio/aac", "audio/x-aac"]),
	("aaf", &["application/octet-stream"]),
	("aam", &["application/x-authorware-map"]),
	("aas", &["application/x-authorware-seg"]),
	("aax", &["audio/vnd.audible.aax"]),
	("abw", &["application/x-abiword"]),
	("ac", &["application/vnd.nokia.n-gage.ac+xml", "application/pkix-attr-cert"]),
	("ac3", &["audio/ac3"]),
	("aca", &["application/octet-stream"]),
	("acc", &["application/vnd.americandynamics.acc"]),
	("accda", &["application/msaccess.addin"]),
	("accdb", &["application/msaccess"]),
	("accdc", &["application/msaccess.cab"]),
	("accde", &["application/msaccess"]),
	("accdr", &["application/msaccess.runtime"]),
	("accdt", &["application/msaccess"]),
	("accdw", &["application/msaccess.webapplication"]),
	("accft", &["application/msaccess.ftemplate"]),
	("ace", &["application/x-ace-compressed"]),
	("acu", &["application/vnd.acucobol"]),
	("acutc", &["application/vnd.acucorp"]),
	("acx", &["application/internet-property-stream"]),
	("addin", &["text/xml"]),
	("ade", &["application/msaccess"]),
	("adobebridge", &["application/x-bridge-url"]),
	("adp", &["application/msaccess", "audio/adpcm"]),
	("adt", &["audio/vnd.dlna.adts"]),
	("adts", &["audio/aac"]),
	("aep", &["application/vnd.audiograph"]),
	("afm", &["application/x-font-type1"]),
	("afp", &["application/vnd.ibm.modcap"]),
	("ahead", &["application/vnd.ahead.space"]),
	("ai", &["application/postscript"]),
	("aif", &["audio/aiff", "audio/x-aiff"]),
	("aifc", &["audio/aiff", "audio/x-aiff"]),
	("aiff", &["audio/aiff", "audio/x-aiff"]),
	("air", &["application/vnd.adobe.air-application-installer-package+zip"]),
	("ait", &["application/vnd.dvb.ait"]),
	("amc", &["application/mpeg"]),
	("ami", &["application/vnd.amiga.ami"]),
	("amr", &["audio/amr"]),
	("anx", &["application/annodex"]),
	("apk", &["application/vnd.android.package-archive"]),
	("apng", &["image/apng"]),
	("appcache", &["text/cache-manifest"]),
	("application", &["application/x-ms-application"]),
	("apr", &["application/vnd.lotus-approach"]),
	("arc", &["application/x-freearc"]),
	("arj", &["application/x-arj"]),
	("art", &["image/x-jg"]),
	("asa", &["application/xml"]),
	("asax", &["application/xml"]),
	("asc", &["application/pgp-signature"]),
	("ascx", &["application/xml"]),
	("asd", &["application/octet-stream"]),
	("asf", &["video/x-ms-asf"]),
	("ashx", &["application/xml"]),
	("asi", &["application/octet-stream"]),
	("asm", &["text/x-asm", "text/plain"]),
	("asmx", &["application/xml"]),
	("aso", &["application/vnd.accpac.simply.aso"]),
	("aspx", &["application/xml"]),
	("asr", &["video/x-ms-asf"]),
	("asx", &["video/x-ms-asf"]),
	("atc", &["application/vnd.acucorp"]),
	("atom", &["application/atom+xml"]),
	("atomcat", &["application/atomcat+xml"]),
	("atomdeleted", &["application/atomdeleted+xml"]),
	("atomsvc", &["application/atomsvc+xml"]),
	("atx", &["application/vnd.antix.game-component"]),
	("au", &["audio/basic"]),
	("avi", &["video/x-msvideo"]),
	("avif", &["image/avif"]),
	("aw", &["application/applixware"]),
	("axa", &["audio/annodex"]),
	("axs", &["application/olescript"]),
	("axv", &["video/annodex"]),
	("azf", &["application/vnd.airzip.filesecure.azf"]),
	("azs", &["application/vnd.airzip.filesecure.azs"]),
	("azv", &["image/vnd.airzip.accelerator.azv"]),
	("azw", &["application/vnd.amazon.ebook"]),
	("b16", &["image/vnd.pco.b16"]),
	("bas", &["text/plain"]),
	("bash", &["application/x-shellscript", "application/x-sh", "text/x-shellscript"]),
	("bat", &["application/x-msdownload"]),
	("bcpio", &["application/x-bcpio"]),
	("bdf", &["application/x-font-bdf"]),
	("bdm", &["application/vnd.syncml.dm+wbxml"]),
	("bdoc", &["application/bdoc"]),
	("bed", &["application/vnd.realvnc.bed"]),
	("bh2", &["application/vnd.fujitsu.oasysprs"]),
	("bin", &["application/octet-stream"]),
	("blb", &["application/x-blorb"]),
	("blorb", &["application/x-blorb"]),
	("bmi", &["application/vnd.bmi"]),
	("bmml", &["application/vnd.balsamiq.bmml+xml"]),
	("bmp", &["image/bmp"]),
	("book", &["application/vnd.framemaker"]),
	("box", &["application/vnd.previewsystems.box"]),
	("boz", &["application/x-bzip2"]),
	("bpk", &["application/octet-stream"]),
	("bsp", &["model/vnd.valve.source.compiled-map"]),
	("btif", &["image/prs.btif"]),
	("buffer", &["application/octet-stream"]),
	("bz", &["application/x-bzip"]),
	("bz2", &["application/x-bzip2"]),
	("c", &["text/x-c", "text/x-csrc"]),
	("c11amc", &["application/vnd.cluetrust.cartomobile-config"]),
	("c11amz", &["application/vnd.cluetrust.cartomobile-config-pkg"]),
	("c4d", &["application/vnd.clonk.c4group"]),
	("c4f", &["application/vnd.clonk.c4group"]),
	("c4g", &["application/vnd.clonk.c4group"]),
	("c4p", &["application/vnd.clonk.c4group"]),
	("c4u", &["application/vnd.clonk.c4group"]),
	("cab", &["application/vnd.ms-cab-compressed"]),
	("caf", &["audio/x-caf"]),
	("calx", &["application/vnd.ms-office.calx"]),
	("cap", &["application/vnd.tcpdump.pcap"]),
	("car", &["application/vnd.curl.car"]),
	("cat", &["application/vnd.ms-pki.seccat"]),
	("cb7", &["application/x-cbr"]),
	("cba", &["application/x-cbr"]),
	("cbr", &["application/x-cbr"]),
	("cbt", &["application/x-cbr"]),
	("cbz", &["application/x-cbr"]),
	("cc", &["text/x-c"]),
	("cco", &["application/x-cocoa"]),
	("cct", &["application/x-director"]),
	("ccxml", &["application/ccxml+xml"]),
	("cd", &["text/plain"]),
	("cdbcmsg", &["application/vnd.contact.cmsg"]),
	("cdda", &["audio/aiff", "audio/x-aiff"]),
	("cdf", &["application/x-cdf", "application/x-netcdf"]),
	("cdfx", &["application/cdfx+xml"]),
	("cdkey", &["application/vnd.mediastation.cdkey"]),
	("cdmia", &["application/cdmi-capability"]),
	("cdmic", &["application/cdmi-container"]),
	("cdmid", &["application/cdmi-domain"]),
	("cdmio", &["application/cdmi-object"]),
	("cdmiq", &["application/cdmi-queue"]),
	("cdx", &["chemical/x-cdx"]),
	("cdxml", &["application/vnd.chemdraw+xml"]),
	("cdy", &["application/vnd.cinderella"]),
	("cer", &["application/pkix-cert", "application/x-x509-ca-cert"]),
	("cfg", &["text/plain"]),
	("cfs", &["application/x-cfs-compressed"]),
	("cgm", &["image/cgm"]),
	("chat", &["application/x-chat"]),
	("chm", &["application/vnd.ms-htmlhelp"]),
	("chrt", &["application/vnd.kde.kchart"]),
	("cif", &["chemical/x-cif"]),
	("cii", &["application/vnd.anser-web-certificate-issue-initiation"]),
	("cil", &["application/vnd.ms-artgalry"]),
	("cjs", &["application/node"]),
	("cla", &["application/vnd.claymore"]),
	("class", &["application/x-java", "application/java-vm", "application/x-java-applet"]),
	("clkk", &["application/vnd.crick.clicker.keyboard"]),
	("clkp", &["application/vnd.crick.clicker.palette"]),
	("clkt", &["application/vnd.crick.clicker.template"]),
	("clkw", &["application/vnd.crick.clicker.wordbank"]),
	("clkx", &["application/vnd.crick.clicker"]),
	("clp", &["application/x-msclip"]),
	("cmc", &["application/vnd.cosmocaller"]),
	("cmd", &["text/plain"]),
	("cmdf", &["chemical/x-cmdf"]),
	("cml", &["chemical/x-cml"]),
	("cmp", &["application/vnd.yellowriver-custom-menu"]),
	("cmx", &["image/x-cmx"]),
	("cnf", &["text/plain"]),
	("cod", &["image/cis-cod", "application/vnd.rim.cod"]),
	("coffee", &["text/coffeescript"]),
	("com", &["application/x-msdownload"]),
	("conf", &["text/plain"]),
	("config", &["application/xml"]),
	("contact", &["text/x-ms-contact"]),
	("coverage", &["application/xml"]),
	("cpio", &["application/x-cpio"]),
	("cpp", &["text/plain", "text/x-c", "text/x-csrc", "text/c++src"]),
	("cpt", &["application/mac-compactpro"]),
	("crd", &["application/x-mscardfile"]),
	("crl", &["application/pkix-crl"]),
	("crt", &["application/x-x509-ca-cert"]),
	("crx", &["application/x-chrome-extension"]),
	("cryptonote", &["application/vnd.rig.cryptonote"]),
	("cs", &["text/plain"]),
	("csdproj", &["text/plain"]),
	("csh", &["application/x-csh"]),
	("csl", &["application/vnd.citationstyles.style+xml"]),
	("csml", &["chemical/x-csml"]),
	("csp", &["application/vnd.commonspace"]),
	("csproj", &["text/plain"]),
	("css", &["text/css"]),
	("cst", &["application/x-director"]),
	("csv", &["text/csv"]),
	("cu", &["application/cu-seeme"]),
	("cur", &["application/octet-stream"]),
	("curl", &["text/vnd.curl"]),
	("cww", &["application/prs.cww"]),
	("cxt", &["application/x-director"]),
	("cxx", &["text/plain", "text/x-c"]),
	("dae", &["model/vnd.collada+xml"]),
	("daf", &["application/vnd.mobius.daf"]),
	("dart", &["application/vnd.dart"]),
	("dat", &["application/octet-stream"]),
	("dataless", &["application/vnd.fdsn.seed"]),
	("datasource", &["application/xml"]),
	("davmount", &["application/davmount+xml"]),
	("dbk", &["application/docbook+xml"]),
	("dbproj", &["text/plain"]),
	("dcr", &["application/x-director"]),
	("dcurl", &["text/vnd.curl.dcurl"]),
	("dd2", &["application/vnd.oma.dd2+xml"]),
	("ddd", &["application/vnd.fujixerox.ddd"]),
	("ddf", &["application/vnd.syncml.dmddf+xml"]),
	("dds", &["image/vnd.ms-dds"]),
	("deb", &["application/vnd.debian.binary-package", "application/x-deb", "application/x-debian-package"]),
	("def", &["text/plain"]),
	("deploy", &["application/octet-stream"]),
	("der", &["application/x-x509-ca-cert"]),
	("dfac", &["application/vnd.dreamfactory"]),
	("dgc", &["application/x-dgc-compressed"]),
	("dgml", &["application/xml"]),
	("dib", &["image/bmp"]),
	("dic", &["text/x-c"]),
	("dif", &["video/x-dv"]),
	("dir", &["application/x-director"]),
	("dis", &["application/vnd.mobius.dis"]),
	("disco", &["text/xml"]),
	("disposition-notification", &["message/disposition-notification"]),
	("dist", &["application/octet-stream"]),
	("distz", &["application/octet-stream"]),
	("divx", &["video/divx"]),
	("djv", &["image/vnd.djvu", "image/vnd.djvu+multipage"]),
	("djvu", &["image/vnd.djvu", "image/vnd.djvu+multipage"]),
	("dll", &["application/x-ms-dos-executable", "application/x-dosexec", "application/x-msdownload"]),
	("dll.config", &["text/xml"]),
	("dlm", &["text/dlm"]),
	("dmg", &["application/octet-stream"]),
	("dmp", &["application/vnd.tcpdump.pcap"]),
	("dms", &["application/octet-stream"]),
	("dna", &["application/vnd.dna"]),
	("doc", &["application/msword"]),
	("docm", &["application/vnd.ms-word.document.macroenabled.12"]),
	("docx", &["application/vnd.openxmlformats-officedocument.wordprocessingml.document"]),
	("dot", &["application/msword"]),
	("dotm", &["application/vnd.ms-word.template.macroenabled.12"]),
	("dotx", &["application/vnd.openxmlformats-officedocument.wordprocessingml.template"]),
	("dp", &["application/vnd.osgi.dp"]),
	("dpg", &["application/vnd.dpgraph"]),
	("dra", &["audio/vnd.dra"]),
	("drle", &["image/dicom-rle"]),
	("dsc", &["text/prs.lines.tag"]),
	("dsp", &["application/octet-stream"]),
	("dssc", &["application/dssc+der"]),
	("dsw", &["text/plain"]),
	("dtb", &["application/x-dtbook+xml"]),
	("dtd", &["application/xml-dtd"]),
	("dts", &["audio/vnd.dts"]),
	("dtsconfig", &["text/xml"]),
	("dtshd", &["audio/vnd.dts.hd"]),
	("dump", &["application/octet-stream"]),
	("dv", &["video/x-dv"]),
	("dvb", &["video/vnd.dvb.file"]),
	("dvi", &["application/x-dvi"]),
	("dwd", &["application/atsc-dwd+xml"]),
	("dwf", &["model/vnd.dwf"]),
	("dwg", &["image/vnd.dwg"]),
	("dwp", &["application/octet-stream"]),
	("dxf", &["image/vnd.dxf"]),
	("dxp", &["application/vnd.spotfire.dxp"]),
	("dxr", &["application/x-director"]),
	("ear", &["application/java-archive"]),
	("ecelp4800", &["audio/vnd.nuera.ecelp4800"]),
	("ecelp7470", &["audio/vnd.nuera.ecelp7470"]),
	("ecelp9600", &["audio/vnd.nuera.ecelp9600"]),
	("ecma", &["application/ecmascript"]),
	("edm", &["application/vnd.novadigm.edm"]),
	("edx", &["application/vnd.novadigm.edx"]),
	("efif", &["application/vnd.picsel"]),
	("ei6", &["application/vnd.pg.osasli"]),
	("elc", &["application/octet-stream"]),
	("emf", &["application/x-msmetafile", "image/emf"]),
	("eml", &["message/rfc822"]),
	("emma", &["application/emma+xml"]),
	("emotionml", &["application/emotionml+xml"]),
	("emz", &["application/x-msmetafile"]),
	("eol", &["audio/vnd.digital-winds"]),
	("eot", &["application/vnd.ms-fontobject"]),
	("eps", &["application/postscript"]),
	("epub", &["application/epub+zip"]),
	("es", &["application/ecmascript"]),
	("es3", &["application/vnd.eszigno3+xml"]),
	("esa", &["application/vnd.osgi.subsystem"]),
	("esf", &["application/vnd.epson.esf"]),
	("et3", &["application/vnd.eszigno3+xml"]),
	("etl", &["application/etl"]),
	("etx", &["text/x-setext"]),
	("eva", &["application/x-eva"]),
	("evy", &["application/envoy", "application/x-envoy"]),
	(
		"exe",
		&["application/vnd.microsoft.portable-executable", "application/x-ms-dos-executable", "application/x-dosexec"]
	),
	("exe.config", &["text/xml"]),
	("exi", &["application/exi"]),
	("ext", &["application/vnd.novadigm.ext"]),
	("ez", &["application/andrew-inset"]),
	("ez2", &["application/vnd.ezpix-album"]),
	("ez3", &["application/vnd.ezpix-package"]),
	("f", &["text/x-fortran"]),
	("f4v", &["video/x-f4v"]),
	("f77", &["text/x-fortran"]),
	("f90", &["text/x-fortran"]),
	("fbs", &["image/vnd.fastbidsheet"]),
	("fcdt", &["application/vnd.adobe.formscentral.fcdt"]),
	("fcs", &["application/vnd.isac.fcs"]),
	("fdf", &["application/vnd.fdf"]),
	("fdt", &["application/fdt+xml"]),
	("fe_launch", &["application/vnd.denovo.fcselayout-link"]),
	("feature", &["text/x-gherkin"]),
	("fg5", &["application/vnd.fujitsu.oasysgp"]),
	("fgd", &["application/x-director"]),
	("fh", &["image/x-freehand"]),
	("fh4", &["image/x-freehand"]),
	("fh5", &["image/x-freehand"]),
	("fh7", &["image/x-freehand"]),
	("fhc", &["image/x-freehand"]),
	("fif", &["application/fractals"]),
	("fig", &["application/x-xfig"]),
	("filters", &["application/xml"]),
	("fish", &["application/x-shellscript", "application/x-sh", "text/x-shellscript"]),
	("fits", &["image/fits"]),
	("fla", &["application/octet-stream"]),
	("flac", &["audio/flac", "audio/x-flac"]),
	("fli", &["video/x-fli"]),
	("flo", &["application/vnd.micrografx.flo"]),
	("flr", &["x-world/x-vrml"]),
	("flv", &["video/x-flv"]),
	("flw", &["application/vnd.kde.kivio"]),
	("flx", &["text/vnd.fmi.flexstor"]),
	("fly", &["text/vnd.fly"]),
	("fm", &["application/vnd.framemaker"]),
	("fnc", &["application/vnd.frogans.fnc"]),
	("fo", &["application/vnd.software602.filler.form+xml"]),
	("for", &["text/x-fortran"]),
	("fpx", &["image/vnd.fpx"]),
	("frame", &["application/vnd.framemaker"]),
	("fsc", &["application/vnd.fsc.weblaunch"]),
	("fsscript", &["application/fsharp-script"]),
	("fst", &["image/vnd.fst"]),
	("fsx", &["application/fsharp-script"]),
	("ftc", &["application/vnd.fluxtime.clip"]),
	("fti", &["application/vnd.anser-web-funds-transfer-initiation"]),
	("fvt", &["video/vnd.fvt"]),
	("fxp", &["application/vnd.adobe.fxp"]),
	("fxpl", &["application/vnd.adobe.fxp"]),
	("fzs", &["application/vnd.fuzzysheet"]),
	("g2w", &["application/vnd.geoplan"]),
	("g3", &["image/g3fax"]),
	("g3w", &["application/vnd.geospace"]),
	("gac", &["application/vnd.groove-account"]),
	("gam", &["application/x-tads"]),
	("gbr", &["application/rpki-ghostbusters"]),
	("gca", &["application/x-gca-compressed"]),
	("gdl", &["model/vnd.gdl"]),
	("gdoc", &["application/vnd.google-apps.document"]),
	("gemini", &["text/gemini"]),
	("generictest", &["application/xml"]),
	("geo", &["application/vnd.dynageo"]),
	("geojson", &["application/geo+json"]),
	("gex", &["application/vnd.geometry-explorer"]),
	("ggb", &["application/vnd.geogebra.file"]),
	("ggt", &["application/vnd.geogebra.tool"]),
	("ghf", &["application/vnd.groove-help"]),
	("gif", &["image/gif"]),
	("gim", &["application/vnd.groove-identity-message"]),
	("glb", &["model/gltf-binary"]),
	("gltf", &["model/gltf+json"]),
	("gmi", &["text/gemini"]),
	("gml", &["application/gml+xml"]),
	("gmx", &["application/vnd.gmx"]),
	("gnumeric", &["application/x-gnumeric"]),
	("gph", &["application/vnd.flographit"]),
	("gpx", &["application/gpx+xml"]),
	("gqf", &["application/vnd.grafeq"]),
	("gqs", &["application/vnd.grafeq"]),
	("gram", &["application/srgs"]),
	("gramps", &["application/x-gramps-xml"]),
	("gre", &["application/vnd.geometry-explorer"]),
	("group", &["text/x-ms-group"]),
	("grv", &["application/vnd.groove-injector"]),
	("grxml", &["application/srgs+xml"]),
	("gsf", &["application/x-font-ghostscript"]),
	("gsheet", &["application/vnd.google-apps.spreadsheet"]),
	("gslides", &["application/vnd.google-apps.presentation"]),
	("gsm", &["audio/x-gsm"]),
	("gtar", &["application/x-gtar"]),
	("gtm", &["application/vnd.groove-tool-message"]),
	("gtw", &["model/vnd.gtw"]),
	("gv", &["text/vnd.graphviz"]),
	("gxf", &["application/gxf"]),
	("gxt", &["application/vnd.geonext"]),
	("gz", &["application/gzip", "application/x-gzip"]),
	("h", &["text/plain"]),
	("h261", &["video/h261"]),
	("h263", &["video/h263"]),
	("h264", &["video/h264"]),
	("hal", &["application/vnd.hal+xml"]),
	("hbci", &["application/vnd.hbci"]),
	("hbs", &["text/x-handlebars-template"]),
	("hdd", &["application/x-virtualbox-hdd"]),
	("hdf", &["application/x-hdf"]),
	("hdml", &["text/x-hdml"]),
	("hdr", &["image/vnd.radiance"]),
	("heic", &["image/heic"]),
	("heics", &["image/heic-sequence"]),
	("heif", &["image/heif"]),
	("heifs", &["image/heif-sequence"]),
	("hej2", &["image/hej2k"]),
	("held", &["application/atsc-held+xml"]),
	("hh", &["text/x-c"]),
	("hhc", &["application/x-oleobject"]),
	("hhk", &["application/octet-stream"]),
	("hhp", &["application/octet-stream"]),
	("hjson", &["application/hjson"]),
	("hlp", &["application/winhlp"]),
	("hpgl", &["application/vnd.hp-hpgl"]),
	("hpid", &["application/vnd.hp-hpid"]),
	("hpp", &["text/plain"]),
	("hps", &["application/vnd.hp-hps"]),
	("hqx", &["application/mac-binhex40"]),
	("hsj2", &["image/hsj2"]),
	("hta", &["application/hta"]),
	("htc", &["text/x-component"]),
	("htke", &["application/vnd.kenameaapp"]),
	("htm", &["text/html"]),
	("html", &["text/html", "application/xhtml+xml"]),
	("htt", &["text/webviewhtml"]),
	("hvd", &["application/vnd.yamaha.hv-dic"]),
	("hvp", &["application/vnd.yamaha.hv-voice"]),
	("hvs", &["application/vnd.yamaha.hv-script"]),
	("hxa", &["application/xml"]),
	("hxc", &["application/xml"]),
	("hxd", &["application/octet-stream"]),
	("hxe", &["application/xml"]),
	("hxf", &["application/xml"]),
	("hxh", &["application/octet-stream"]),
	("hxi", &["application/octet-stream"]),
	("hxk", &["application/xml"]),
	("hxq", &["application/octet-stream"]),
	("hxr", &["application/octet-stream"]),
	("hxs", &["application/octet-stream"]),
	("hxt", &["text/html"]),
	("hxv", &["application/xml"]),
	("hxw", &["application/octet-stream"]),
	("hxx", &["text/plain"]),
	("i", &["text/plain"]),
	("i2g", &["application/vnd.intergeo"]),
	("icc", &["application/vnd.iccprofile"]),
	("ice", &["x-conference/x-cooltalk"]),
	("icm", &["application/vnd.iccprofile"]),
	("ico", &["image/vnd.microsoft.icon", "image/x-icon"]),
	("ics", &["text/calendar"]),
	("idl", &["text/plain"]),
	("ief", &["image/ief"]),
	("ifb", &["text/calendar"]),
	("ifm", &["application/vnd.shana.informed.formdata"]),
	("iges", &["model/iges"]),
	("igl", &["application/vnd.igloader"]),
	("igm", &["application/vnd.insors.igm"]),
	("igs", &["model/iges"]),
	("igx", &["application/vnd.micrografx.igx"]),
	("iif", &["application/vnd.shana.informed.interchange"]),
	("iii", &["application/x-iphone"]),
	("img", &["application/octet-stream"]),
	("imp", &["application/vnd.accpac.simply.imp"]),
	("ims", &["application/vnd.ms-ims"]),
	("in", &["text/plain", "application/xml"]),
	("inc", &["text/plain"]),
	("inf", &["application/octet-stream"]),
	("ini", &["text/plain"]),
	("ink", &["application/inkml+xml"]),
	("inkml", &["application/inkml+xml"]),
	("inl", &["text/plain"]),
	("ins", &["application/x-internet-signup"]),
	("install", &["application/x-install-instructions"]),
	("iota", &["application/vnd.astraea-software.iota"]),
	("ipa", &["application/x-itunes-ipa"]),
	("ipfix", &["application/ipfix"]),
	("ipg", &["application/x-itunes-ipg"]),
	("ipk", &["application/vnd.shana.informed.package"]),
	("ipproj", &["text/plain"]),
	("ipsw", &["application/x-itunes-ipsw"]),
	("iqy", &["text/x-ms-iqy"]),
	("irm", &["application/vnd.ibm.rights-management"]),
	("irp", &["application/vnd.irepository.package+xml"]),
	("iso", &["application/x-iso9660-image"]),
	("isp", &["application/x-internet-signup"]),
	("ite", &["application/x-itunes-ite"]),
	("itlp", &["application/x-itunes-itlp"]),
	("itms", &["application/x-itunes-itms"]),
	("itp", &["application/vnd.shana.informed.formtemplate"]),
	("itpc", &["application/x-itunes-itpc"]),
	("its", &["application/its+xml"]),
	("ivf", &["video/x-ivf"]),
	("ivp", &["application/vnd.immervision-ivp"]),
	("ivu", &["application/vnd.immervision-ivu"]),
	("jad", &["text/vnd.sun.j2me.app-descriptor"]),
	("jade", &["text/jade"]),
	("jam", &["application/vnd.jam"]),
	("jar", &["application/java-archive"]),
	("jardiff", &["application/x-java-archive-diff"]),
	("java", &["text/x-java-source", "application/octet-stream"]),
	("jck", &["application/liquidmotion"]),
	("jcz", &["application/liquidmotion"]),
	("jfif", &["image/jpeg"]),
	("jisp", &["application/vnd.jisp"]),
	("jls", &["image/jls"]),
	("jlt", &["application/vnd.hp-jlyt"]),
	("jng", &["image/x-jng"]),
	("jnlp", &["application/x-java-jnlp-file"]),
	("joda", &["application/vnd.joost.joda-archive"]),
	("jp2", &["image/jp2"]),
	("jpb", &["application/octet-stream"]),
	("jpe", &["image/jpeg"]),
	("jpeg", &["image/jpeg"]),
	("jpf", &["image/jpx"]),
	("jpg", &["image/jpeg"]),
	("jpg2", &["image/jp2"]),
	("jpgm", &["video/jpm"]),
	("jpgv", &["video/jpeg"]),
	("jph", &["image/jph"]),
	("jpm", &["image/jpm"]),
	("jpx", &["image/jpx"]),
	("js", &["application/javascript"]),
	("jsm", &["application/javascript"]),
	("json", &["application/json"]),
	("json5", &["application/json5"]),
	("jsonld", &["application/ld+json"]),
	("jsonml", &["application/jsonml+json"]),
	("jsx", &["text/jscript", "text/jsx"]),
	("jsxbin", &["text/plain"]),
	("jxr", &["image/jxr"]),
	("jxra", &["image/jxra"]),
	("jxrs", &["image/jxrs"]),
	("jxs", &["image/jxs"]),
	("jxsc", &["image/jxsc"]),
	("jxsi", &["image/jxsi"]),
	("jxss", &["image/jxss"]),
	("kar", &["audio/midi"]),
	("karbon", &["application/vnd.kde.karbon"]),
	("kdbx", &["application/x-keepass2"]),
	("key", &["application/vnd.apple.keynote"]),
	("kfo", &["application/vnd.kde.kformula"]),
	("kia", &["application/vnd.kidspiration"]),
	("kml", &["application/vnd.google-earth.kml+xml"]),
	("kmz", &["application/vnd.google-earth.kmz"]),
	("kne", &["application/vnd.kinar"]),
	("knp", &["application/vnd.kinar"]),
	("kon", &["application/vnd.kde.kontour"]),
	("kpr", &["application/vnd.kde.kpresenter"]),
	("kpt", &["application/vnd.kde.kpresenter"]),
	("kpxx", &["application/vnd.ds-keypoint"]),
	("kra", &["application/x-krita"]),
	("ksp", &["application/vnd.kde.kspread"]),
	("ktr", &["application/vnd.kahootz"]),
	("ktx", &["image/ktx"]),
	("ktx2", &["image/ktx2"]),
	("ktz", &["application/vnd.kahootz"]),
	("kwd", &["application/vnd.kde.kword"]),
	("kwt", &["application/vnd.kde.kword"]),
	("lasxml", &["application/vnd.las.las+xml"]),
	("latex", &["application/x-latex"]),
	("lbd", &["application/vnd.llamagraphics.life-balance.desktop"]),
	("lbe", &["application/vnd.llamagraphics.life-balance.exchange+xml"]),
	("les", &["application/vnd.hhe.lesson-player"]),
	("less", &["text/less"]),
	("lgr", &["application/lgr+xml"]),
	("lha", &["application/x-lzh-compressed"]),
	("library-ms", &["application/windows-library+xml"]),
	("link66", &["application/vnd.route66.link66+xml"]),
	("list", &["text/plain"]),
	("list3820", &["application/vnd.ibm.modcap"]),
	("listafp", &["application/vnd.ibm.modcap"]),
	("lit", &["application/x-ms-reader"]),
	("litcoffee", &["text/coffeescript"]),
	("lnk", &["application/x-ms-shortcut"]),
	("loadtest", &["application/xml"]),
	("log", &["text/plain"]),
	("lostxml", &["application/lost+xml"]),
	("lpk", &["application/octet-stream"]),
	("lrf", &["application/octet-stream"]),
	("lrm", &["application/vnd.ms-lrm"]),
	("lsf", &["video/x-la-asf"]),
	("lst", &["text/plain"]),
	("lsx", &["video/x-la-asf"]),
	("ltf", &["application/vnd.frogans.ltf"]),
	("lua", &["text/x-lua"]),
	("luac", &["application/x-lua-bytecode"]),
	("lvp", &["audio/vnd.lucent.voice"]),
	("lwp", &["application/vnd.lotus-wordpro"]),
	("lzh", &["application/x-lzh-compressed"]),
	("m13", &["application/x-msmediaview"]),
	("m14", &["application/x-msmediaview"]),
	("m1v", &["video/mpeg"]),
	("m21", &["application/mp21"]),
	("m2a", &["audio/mpeg"]),
	("m2t", &["video/vnd.dlna.mpeg-tts"]),
	("m2ts", &["video/vnd.dlna.mpeg-tts"]),
	("m2v", &["video/mpeg"]),
	("m3a", &["audio/mpeg"]),
	("m3u", &["audio/x-mpegurl", "application/vnd.apple.mpegurl"]),
	("m3u8", &["audio/x-mpegurl"]),
	("m4a", &["audio/m4a", "audio/mp4"]),
	("m4b", &["audio/m4b"]),
	("m4p", &["audio/m4p", "audio/mp4", "application/mp4"]),
	("m4r", &["audio/x-m4r"]),
	("m4s", &["video/iso.segment"]),
	("m4u", &["video/vnd.mpegurl"]),
	("m4v", &["video/x-m4v"]),
	("ma", &["application/mathematica"]),
	("mac", &["image/x-macpaint"]),
	("mads", &["application/mads+xml"]),
	("maei", &["application/mmt-aei+xml"]),
	("mag", &["application/vnd.ecowin.chart"]),
	("mak", &["text/plain"]),
	("maker", &["application/vnd.framemaker"]),
	("man", &["text/troff", "application/x-troff-man"]),
	("manifest", &["text/cache-manifest", "application/x-ms-manifest"]),
	("map", &["application/json", "text/plain"]),
	("mar", &["application/octet-stream"]),
	("markdown", &["text/markdown"]),
	("master", &["application/xml"]),
	("mathml", &["application/mathml+xml"]),
	("mb", &["application/mathematica"]),
	("mbk", &["application/vnd.mobius.mbk"]),
	("mbox", &["application/mbox"]),
	("mc1", &["application/vnd.medcalcdata"]),
	("mcd", &["application/vnd.mcd"]),
	("mcurl", &["text/vnd.curl.mcurl"]),
	("md", &["text/markdown", "text/x-markdown"]),
	("mda", &["application/msaccess"]),
	("mdb", &["application/x-msaccess"]),
	("mde", &["application/msaccess"]),
	("mdi", &["image/vnd.ms-modi"]),
	("mdp", &["application/octet-stream"]),
	("mdx", &["text/mdx"]),
	("me", &["text/troff", "application/x-troff-me"]),
	("mesh", &["model/mesh"]),
	("meta4", &["application/metalink4+xml"]),
	("metalink", &["application/metalink+xml"]),
	("mets", &["application/mets+xml"]),
	("mfm", &["application/vnd.mfmp"]),
	("mfp", &["application/x-shockwave-flash"]),
	("mft", &["application/rpki-manifest"]),
	("mgp", &["application/vnd.osgeo.mapguide.package"]),
	("mgz", &["application/vnd.proteus.magazine"]),
	("mht", &["message/rfc822"]),
	("mhtml", &["message/rfc822"]),
	("mid", &["audio/midi", "audio/mid"]),
	("midi", &["audio/midi", "audio/mid"]),
	("mie", &["application/x-mie"]),
	("mif", &["application/vnd.mif"]),
	("mime", &["message/rfc822"]),
	("mix", &["application/octet-stream"]),
	("mj2", &["video/mj2"]),
	("mjp2", &["video/mj2"]),
	("mjs", &["application/javascript"]),
	("mk", &["text/plain"]),
	("mk3d", &["video/x-matroska"]),
	("mka", &["audio/x-matroska"]),
	("mkd", &["text/x-markdown"]),
	("mks", &["video/x-matroska"]),
	("mkv", &["video/x-matroska"]),
	("mlp", &["application/vnd.dolby.mlp"]),
	("mmd", &["application/vnd.chipnuts.karaoke-mmd"]),
	("mmf", &["application/vnd.smaf", "application/x-smaf"]),
	("mml", &["text/mathml"]),
	("mmr", &["image/vnd.fujixerox.edmics-mmr"]),
	("mng", &["video/x-mng"]),
	("mno", &["text/xml"]),
	("mny", &["application/x-msmoney"]),
	("mobi", &["application/x-mobipocket-ebook"]),
	("mod", &["video/mpeg"]),
	("mods", &["application/mods+xml"]),
	("mov", &["video/quicktime"]),
	("movie", &["video/x-sgi-movie"]),
	("mp2", &["audio/mpeg", "video/mpeg"]),
	("mp21", &["application/mp21"]),
	("mp2a", &["audio/mpeg"]),
	("mp2v", &["video/mpeg"]),
	("mp3", &["audio/mpeg"]),
	("mp4", &["video/mp4"]),
	("mp4a", &["audio/mp4"]),
	("mp4s", &["application/mp4"]),
	("mp4v", &["video/mp4"]),
	("mpa", &["video/mpeg"]),
	("mpc", &["application/vnd.mophun.certificate"]),
	("mpd", &["application/dash+xml"]),
	("mpe", &["video/mpeg"]),
	("mpeg", &["video/mpeg"]),
	("mpf", &["application/vnd.ms-mediapackage"]),
	("mpg", &["video/mpeg"]),
	("mpg4", &["video/mp4"]),
	("mpga", &["audio/mpeg"]),
	("mpkg", &["application/vnd.apple.installer+xml"]),
	("mpm", &["application/vnd.blueice.multipass"]),
	("mpn", &["application/vnd.mophun.application"]),
	("mpp", &["application/vnd.ms-project"]),
	("mpt", &["application/vnd.ms-project"]),
	("mpv2", &["video/mpeg"]),
	("mpy", &["application/vnd.ibm.minipay"]),
	("mqv", &["video/quicktime"]),
	("mqy", &["application/vnd.mobius.mqy"]),
	("mrc", &["application/marc"]),
	("mrcx", &["application/marcxml+xml"]),
	("ms", &["text/troff", "application/x-troff-ms"]),
	("mscml", &["application/mediaservercontrol+xml"]),
	("mseed", &["application/vnd.fdsn.mseed"]),
	("mseq", &["application/vnd.mseq"]),
	("msf", &["application/vnd.epson.msf"]),
	("msg", &["application/vnd.ms-outlook"]),
	("msh", &["model/mesh"]),
	("msi", &["application/x-msi"]),
	("msl", &["application/vnd.mobius.msl"]),
	("msm", &["application/octet-stream"]),
	("mso", &["application/octet-stream"]),
	("msp", &["application/octet-stream"]),
	("msty", &["application/vnd.muvee.style"]),
	("mtl", &["model/mtl"]),
	("mts", &["model/vnd.mts", "video/vnd.dlna.mpeg-tts"]),
	("mtx", &["application/xml"]),
	("mus", &["application/vnd.musician"]),
	("musd", &["application/mmt-usd+xml"]),
	("musicxml", &["application/vnd.recordare.musicxml+xml"]),
	("mvb", &["application/x-msmediaview"]),
	("mvc", &["application/x-miva-compiled"]),
	("mwf", &["application/vnd.mfer"]),
	("mxf", &["application/mxf"]),
	("mxl", &["application/vnd.recordare.musicxml"]),
	("mxml", &["aduio/mobile-xmf", "application/xv+xml"]),
	("mxp", &["application/x-mmxp"]),
	("mxs", &["application/vnd.triscape.mxs"]),
	("mxu", &["video/vnd.mpegurl"]),
	("n-gage", &["application/vnd.nokia.n-gage.symbian.install"]),
	("n3", &["text/n3"]),
	("nb", &["application/mathematica"]),
	("nbp", &["application/vnd.wolfram.player"]),
	("nc", &["application/x-netcdf"]),
	("ncx", &["application/x-dtbncx+xml"]),
	("nfo", &["text/x-nfo"]),
	("ngdat", &["application/vnd.nokia.n-gage.data"]),
	("nitf", &["application/vnd.nitf"]),
	("nlu", &["application/vnd.neurolanguage.nlu"]),
	("nml", &["application/vnd.enliven"]),
	("nnd", &["application/vnd.noblenet-directory"]),
	("nns", &["application/vnd.noblenet-sealer"]),
	("nnw", &["application/vnd.noblenet-web"]),
	("npx", &["image/vnd.net-fpx"]),
	("nq", &["application/x-conference", "application/n-quads"]),
	("nsc", &["video/x-ms-asf"]),
	("nsf", &["application/vnd.lotus-notes"]),
	("nt", &["application/n-triples"]),
	("ntf", &["application/vnd.nitf"]),
	("numbers", &["application/vnd.apple.numbers"]),
	("nws", &["message/rfc822"]),
	("nzb", &["application/x-nzb"]),
	("oa2", &["application/vnd.fujitsu.oasys2"]),
	("oa3", &["application/vnd.fujitsu.oasys3"]),
	("oas", &["application/vnd.fujitsu.oasys"]),
	("obd", &["application/x-msbinder"]),
	("obgx", &["application/vnd.openblox.game+xml"]),
	("obj", &["model/obj", "application/x-tgif"]),
	("ocx", &["application/octet-stream"]),
	("oda", &["application/oda"]),
	("odb", &["application/vnd.oasis.opendocument.database"]),
	("odc", &["application/vnd.oasis.opendocument.chart"]),
	("odf", &["application/vnd.oasis.opendocument.formula"]),
	("odft", &["application/vnd.oasis.opendocument.formula-template"]),
	("odg", &["application/vnd.oasis.opendocument.graphics"]),
	("odh", &["text/plain"]),
	("odi", &["application/vnd.oasis.opendocument.image"]),
	("odl", &["text/plain"]),
	("odm", &["application/vnd.oasis.opendocument.text-master"]),
	("odp", &["application/vnd.oasis.opendocument.presentation"]),
	("ods", &["application/vnd.oasis.opendocument.spreadsheet"]),
	("odt", &["application/vnd.oasis.opendocument.text"]),
	("oga", &["audio/ogg", "audio/x-vorbis+ogg"]),
	("ogex", &["model/vnd.opengex"]),
	("ogg", &["audio/ogg", "audio/x-vorbis+ogg"]),
	("ogv", &["video/ogg", "video/x-theora+ogg"]),
	("ogx", &["application/ogg"]),
	("omdoc", &["application/omdoc+xml"]),
	("one", &["application/onenote"]),
	("onea", &["application/onenote"]),
	("onepkg", &["application/onenote"]),
	("onetmp", &["application/onenote"]),
	("onetoc", &["application/onenote"]),
	("onetoc2", &["application/onenote"]),
	("opf", &["application/oebps-package+xml"]),
	("opml", &["text/x-opml"]),
	("oprc", &["application/vnd.palm"]),
	("opus", &["audio/ogg", "audio/x-opus+ogg"]),
	("orderedtest", &["application/xml"]),
	("org", &["application/vnd.lotus-organizer"]),
	("osdx", &["application/opensearchdescription+xml"]),
	("osf", &["application/vnd.yamaha.openscoreformat"]),
	("osfpvg", &["application/vnd.yamaha.openscoreformat.osfpvg+xml"]),
	("osm", &["application/vnd.openstreetmap.data+xml"]),
	("otc", &["application/vnd.oasis.opendocument.chart-template"]),
	("otf", &["font/otf", "application/font-sfnt"]),
	("otg", &["application/vnd.oasis.opendocument.graphics-template"]),
	("oth", &["application/vnd.oasis.opendocument.text-web"]),
	("oti", &["application/vnd.oasis.opendocument.image-template"]),
	("otp", &["application/vnd.oasis.opendocument.presentation-template"]),
	("ots", &["application/vnd.oasis.opendocument.spreadsheet-template"]),
	("ott", &["application/vnd.oasis.opendocument.text-template"]),
	("ova", &["application/x-virtualbox-ova"]),
	("ovf", &["application/x-virtualbox-ovf"]),
	("owl", &["application/rdf+xml"]),
	("oxps", &["application/oxps"]),
	("oxt", &["application/vnd.openofficeorg.extension"]),
	("p", &["text/x-pascal"]),
	("p10", &["application/pkcs10"]),
	("p12", &["application/x-pkcs12"]),
	("p7b", &["application/x-pkcs7-certificates"]),
	("p7c", &["application/pkcs7-mime"]),
	("p7m", &["application/pkcs7-mime"]),
	("p7r", &["application/x-pkcs7-certreqresp"]),
	("p7s", &["application/pkcs7-signature"]),
	("p8", &["application/pkcs8"]),
	("pac", &["application/x-ns-proxy-autoconfig"]),
	("pages", &["application/vnd.apple.pages"]),
	("pas", &["text/x-pascal"]),
	("paw", &["application/vnd.pawaafile"]),
	("pbd", &["application/vnd.powerbuilder6"]),
	("pbm", &["image/x-portable-bitmap"]),
	("pcap", &["application/vnd.tcpdump.pcap"]),
	("pcast", &["application/x-podcast"]),
	("pcf", &["application/x-font-pcf"]),
	("pcl", &["application/vnd.hp-pcl"]),
	("pclxl", &["application/vnd.hp-pclxl"]),
	("pct", &["image/pict", "image/x-pict"]),
	("pcurl", &["application/vnd.curl.pcurl"]),
	("pcx", &["image/vnd.zbrush.pcx"]),
	("pcz", &["application/octet-stream"]),
	("pdb", &["application/vnd.palm"]),
	("pde", &["text/x-processing"]),
	("pdf", &["application/pdf"]),
	("pem", &["application/x-x509-ca-cert"]),
	("pfa", &["application/x-font-type1"]),
	("pfb", &["application/x-font-type1"]),
	("pfm", &["application/x-font-type1"]),
	("pfr", &["application/font-tdpfr"]),
	("pfx", &["application/x-pkcs12"]),
	("pgm", &["image/x-portable-graymap"]),
	("pgn", &["application/x-chess-pgn"]),
	("pgp", &["application/pgp-encrypted"]),
	("php", &["application/x-php", "application/x-httpd-php"]),
	("pic", &["image/pict", "image/x-pict"]),
	("pict", &["image/pict", "image/x-pict"]),
	("pkg", &["application/octet-stream"]),
	("pkgdef", &["text/plain"]),
	("pkgundef", &["text/plain"]),
	("pki", &["application/pkixcmp"]),
	("pkipath", &["application/pkix-pkipath"]),
	("pko", &["application/vnd.ms-pki.pko"]),
	("pkpass", &["application/vnd.apple.pkpass"]),
	("pl", &["application/x-perl"]),
	("plb", &["application/vnd.3gpp.pic-bw-large"]),
	("plc", &["application/vnd.mobius.plc"]),
	("plf", &["application/vnd.pocketlearn"]),
	("pls", &["audio/scpls", "application/pls+xml"]),
	("pm", &["application/x-perl"]),
	("pma", &["application/x-perfmon"]),
	("pmc", &["application/x-perfmon"]),
	("pml", &["application/x-perfmon", "application/vnd.ctc-posml"]),
	("pmr", &["application/x-perfmon"]),
	("pmw", &["application/x-perfmon"]),
	("png", &["image/png"]),
	("pnm", &["image/x-portable-anymap"]),
	("pnt", &["image/x-macpaint"]),
	("pntg", &["image/x-macpaint"]),
	("pnz", &["image/png"]),
	("portpkg", &["application/vnd.macports.portpkg"]),
	("pot", &["application/vnd.ms-powerpoint"]),
	("potm", &["application/vnd.ms-powerpoint.template.macroenabled.12"]),
	("potx", &["application/vnd.openxmlformats-officedocument.presentationml.template"]),
	("ppa", &["application/vnd.ms-powerpoint"]),
	("ppam", &["application/vnd.ms-powerpoint.addin.macroenabled.12"]),
	("ppd", &["application/vnd.cups-ppd"]),
	("ppm", &["image/x-portable-pixmap"]),
	("pps", &["application/vnd.ms-powerpoint"]),
	("ppsm", &["application/vnd.ms-powerpoint.slideshow.macroenabled.12"]),
	("ppsx", &["application/vnd.openxmlformats-officedocument.presentationml.slideshow"]),
	("ppt", &["application/vnd.ms-powerpoint"]),
	("pptm", &["application/vnd.ms-powerpoint.presentation.macroenabled.12"]),
	("pptx", &["application/vnd.openxmlformats-officedocument.presentationml.presentation"]),
	("pqa", &["application/vnd.palm"]),
	("prc", &["application/x-mobipocket-ebook"]),
	("pre", &["application/vnd.lotus-freelance"]),
	("prf", &["application/pics-rules"]),
	("prm", &["application/octet-stream"]),
	("provx", &["application/provenance+xml"]),
	("prx", &["application/octet-stream"]),
	("ps", &["application/postscript"]),
	("psb", &["application/vnd.3gpp.pic-bw-small"]),
	("psc1", &["application/PowerShell"]),
	("psd", &["image/vnd.adobe.photoshop"]),
	("psess", &["application/xml"]),
	("psf", &["application/x-font-linux-psf"]),
	("pskcxml", &["application/pskc+xml"]),
	("psm", &["application/octet-stream"]),
	("psp", &["application/octet-stream"]),
	("pst", &["application/vnd.ms-outlook"]),
	("pti", &["image/prs.pti"]),
	("ptid", &["application/vnd.pvi.ptid1"]),
	("pub", &["application/x-mspublisher"]),
	("pvb", &["application/vnd.3gpp.pic-bw-var"]),
	("pwn", &["application/vnd.3m.post-it-notes"]),
	("pwz", &["application/vnd.ms-powerpoint"]),
	("py", &["text/plain"]),
	("pya", &["audio/vnd.ms-playready.media.pya"]),
	("pyv", &["video/vnd.ms-playready.media.pyv"]),
	("qam", &["application/vnd.epson.quickanime"]),
	("qbo", &["application/vnd.intu.qbo"]),
	("qfx", &["application/vnd.intu.qfx"]),
	("qht", &["text/x-html-insertion"]),
	("qhtm", &["text/x-html-insertion"]),
	("qps", &["application/vnd.publishare-delta-tree"]),
	("qt", &["video/quicktime"]),
	("qti", &["image/x-quicktime"]),
	("qtif", &["image/x-quicktime"]),
	("qtl", &["application/x-quicktimeplayer"]),
	("qwd", &["application/vnd.quark.quarkxpress"]),
	("qwt", &["application/vnd.quark.quarkxpress"]),
	("qxb", &["application/vnd.quark.quarkxpress"]),
	("qxd", &["application/octet-stream"]),
	("qxl", &["application/vnd.quark.quarkxpress"]),
	("qxt", &["application/vnd.quark.quarkxpress"]),
	("ra", &["audio/x-pn-realaudio"]),
	("ram", &["audio/x-pn-realaudio"]),
	("raml", &["application/raml+yaml"]),
	("rapd", &["application/route-apd+xml"]),
	("rar", &["application/vnd.rar", "application/x-rar", "application/x-rar-compressed"]),
	("ras", &["image/x-cmu-raster"]),
	("rat", &["application/rat-file"]),
	("rc", &["text/plain"]),
	("rc2", &["text/plain"]),
	("rcprofile", &["application/vnd.ipunplugged.rcprofile"]),
	("rct", &["text/plain"]),
	("rdf", &["application/rdf+xml"]),
	("rdlc", &["application/xml"]),
	("rdz", &["application/vnd.data-vision.rdz"]),
	("reg", &["text/plain"]),
	("relo", &["application/p2p-overlay+xml"]),
	("rep", &["application/vnd.businessobjects"]),
	("res", &["application/x-dtbresource+xml"]),
	("resx", &["application/xml"]),
	("rf", &["image/vnd.rn-realflash"]),
	("rgb", &["image/x-rgb"]),
	("rgs", &["text/plain"]),
	("rif", &["application/reginfo+xml"]),
	("rip", &["audio/vnd.rip"]),
	("ris", &["application/x-research-info-systems"]),
	("rl", &["application/resource-lists+xml"]),
	("rlc", &["image/vnd.fujixerox.edmics-rlc"]),
	("rld", &["application/resource-lists-diff+xml"]),
	("rm", &["application/vnd.rn-realmedia"]),
	("rmi", &["audio/midi", "audio/mid"]),
	("rmp", &["audio/x-pn-realaudio-plugin", "application/vnd.rn-rn_music_package"]),
	("rms", &["application/vnd.jcp.javame.midlet-rms"]),
	("rmvb", &["application/vnd.rn-realmedia-vbr"]),
	("rnc", &["application/relax-ng-compact-syntax"]),
	("rng", &["application/xml"]),
	("roa", &["application/rpki-roa"]),
	("roff", &["text/troff"]),
	("rp9", &["application/vnd.cloanto.rp9"]),
	("rpa", &["application/x-rpa"]),
	("rpm", &["application/x-rpm", "application/x-redhat-package-manager", "audio/x-pn-realaudio-plugin"]),
	("rpss", &["application/vnd.nokia.radio-presets"]),
	("rpst", &["application/vnd.nokia.radio-preset"]),
	("rq", &["application/sparql-query"]),
	("rqy", &["text/x-ms-rqy"]),
	("rs", &["text/x-rust", "application/rls-services+xml"]),
	("rsat", &["application/atsc-rsat+xml"]),
	("rsd", &["application/rsd+xml"]),
	("rsheet", &["application/urc-ressheet+xml"]),
	("rss", &["application/rss+xml"]),
	("rtf", &["text/rtf", "application/rtf"]),
	("rtx", &["text/richtext"]),
	("ruleset", &["application/xml"]),
	("run", &["application/x-makeself"]),
	("rusd", &["application/route-usd+xml"]),
	("rvt", &["application/octet-stream"]),
	("s", &["text/x-asm"]),
	("s3m", &["audio/s3m"]),
	("saf", &["application/vnd.yamaha.smaf-audio"]),
	("safariextz", &["application/x-safari-safariextz"]),
	("sass", &["text/x-sass"]),
	("sbml", &["application/sbml+xml"]),
	("sc", &["application/vnd.ibm.secure-container"]),
	("scd", &["application/x-msschedule"]),
	("scm", &["application/vnd.lotus-screencam"]),
	("scq", &["application/scvp-cv-request"]),
	(
		"scr",
		&["application/vnd.microsoft.portable-executable", "application/x-ms-dos-executable", "application/x-dosexec",
			"text/plain"]
	),
	("scs", &["application/scvp-cv-response"]),
	("scss", &["text/x-scss"]),
	("sct", &["text/scriptlet"]),
	("scurl", &["text/vnd.curl.scurl"]),
	("sd2", &["audio/x-sd2"]),
	("sda", &["application/vnd.stardivision.draw"]),
	("sdc", &["application/vnd.stardivision.calc"]),
	("sdd", &["application/vnd.stardivision.impress"]),
	("sdkd", &["application/vnd.solent.sdkm+xml"]),
	("sdkm", &["application/vnd.solent.sdkm+xml"]),
	("sdp", &["application/sdp"]),
	("sdw", &["application/vnd.stardivision.writer"]),
	("sea", &["application/x-sea"]),
	("searchconnector-ms", &["application/windows-search-connector+xml"]),
	("see", &["application/vnd.seemail"]),
	("seed", &["application/vnd.fdsn.seed"]),
	("sema", &["application/vnd.sema"]),
	("semd", &["application/vnd.semd"]),
	("semf", &["application/vnd.semf"]),
	("senmlx", &["application/senml+xml"]),
	("sensmlx", &["application/sensml+xml"]),
	("ser", &["application/java-serialized-object"]),
	("setpay", &["application/set-payment-initiation"]),
	("setreg", &["application/set-registration-initiation"]),
	("settings", &["application/xml"]),
	("sfd-hdstx", &["application/vnd.hydrostatix.sof-data"]),
	("sfs", &["application/vnd.spotfire.sfs"]),
	("sfv", &["text/x-sfv"]),
	("sgi", &["image/sgi"]),
	("sgimb", &["application/x-sgimb"]),
	("sgl", &["application/vnd.stardivision.writer-global"]),
	("sgm", &["text/sgml"]),
	("sgml", &["text/sgml"]),
	("sh", &["application/x-shellscript", "application/x-sh", "text/x-shellscript"]),
	("shar", &["application/x-shar"]),
	("shex", &["text/shex"]),
	("shf", &["application/shf+xml"]),
	("shtml", &["text/html"]),
	("sid", &["image/x-mrsid-image"]),
	("sieve", &["application/sieve"]),
	("sig", &["application/pgp-signature"]),
	("sil", &["audio/silk"]),
	("silo", &["model/mesh"]),
	("sis", &["application/vnd.symbian.install"]),
	("sisx", &["application/vnd.symbian.install"]),
	("sit", &["application/x-stuffit"]),
	("sitemap", &["application/xml"]),
	("sitx", &["application/x-stuffitx"]),
	("siv", &["application/sieve"]),
	("skd", &["application/vnd.koan"]),
	("skin", &["application/xml"]),
	("skm", &["application/vnd.koan"]),
	("skp", &["application/vnd.koan"]),
	("skt", &["application/vnd.koan"]),
	("sldm", &["application/vnd.ms-powerpoint.slide.macroenabled.12"]),
	("sldx", &["application/vnd.openxmlformats-officedocument.presentationml.slide"]),
	("slim", &["text/slim"]),
	("slk", &["application/vnd.ms-excel"]),
	("slm", &["text/slim"]),
	("sln", &["text/plain"]),
	("sls", &["application/route-s-tsid+xml"]),
	("slt", &["application/vnd.epson.salt"]),
	("slupkg-ms", &["application/x-ms-license"]),
	("sm", &["application/vnd.stepmania.stepchart"]),
	("smd", &["audio/x-smd"]),
	("smf", &["application/vnd.stardivision.math"]),
	("smi", &["application/octet-stream"]),
	("smil", &["application/smil+xml"]),
	("smv", &["video/x-smv"]),
	("smx", &["audio/x-smd"]),
	("smz", &["audio/x-smd"]),
	("smzip", &["application/vnd.stepmania.package"]),
	("snd", &["audio/basic"]),
	("snf", &["application/x-font-snf"]),
	("snippet", &["application/xml"]),
	("snp", &["application/octet-stream"]),
	("so", &["application/octet-stream"]),
	("sol", &["text/plain"]),
	("sor", &["text/plain"]),
	("spc", &["application/x-pkcs7-certificates"]),
	("spdx", &["text/spdx"]),
	("spf", &["application/vnd.yamaha.smaf-phrase"]),
	("spl", &["application/futuresplash", "application/x-futuresplash"]),
	("spot", &["text/vnd.in3d.spot"]),
	("spp", &["application/scvp-vp-response"]),
	("spq", &["application/scvp-vp-request"]),
	("spx", &["audio/ogg", "audio/x-speex+ogg"]),
	("sql", &["application/x-sql"]),
	("src", &["application/x-wais-source"]),
	("srf", &["text/plain"]),
	("srt", &["application/x-subrip"]),
	("sru", &["application/sru+xml"]),
	("srx", &["application/sparql-results+xml"]),
	("ssdl", &["application/ssdl+xml"]),
	("sse", &["application/vnd.kodak-descriptor"]),
	("ssf", &["application/vnd.epson.ssf"]),
	("ssisdeploymentmanifest", &["text/xml"]),
	("ssm", &["application/streamingmedia"]),
	("ssml", &["application/ssml+xml"]),
	("sst", &["application/vnd.ms-pki.certstore"]),
	("st", &["application/vnd.sailingtracker.track"]),
	("stc", &["application/vnd.sun.xml.calc.template"]),
	("std", &["application/vnd.sun.xml.draw.template"]),
	("step", &["application/step"]),
	("stf", &["application/vnd.wt.stf"]),
	("sti", &["application/vnd.sun.xml.impress.template"]),
	("stk", &["application/hyperstudio"]),
	("stl", &["model/stl", "application/vnd.ms-pki.stl"]),
	("stp", &["application/step"]),
	("str", &["application/vnd.pg.format"]),
	("stw", &["application/vnd.sun.xml.writer.template"]),
	("styl", &["text/stylus"]),
	("stylus", &["text/stylus"]),
	("sub", &["text/vnd.dvb.subtitle"]),
	("sus", &["application/vnd.sus-calendar"]),
	("susp", &["application/vnd.sus-calendar"]),
	("sv4cpio", &["application/x-sv4cpio"]),
	("sv4crc", &["application/x-sv4crc"]),
	("svc", &["application/vnd.dvb.service"]),
	("svd", &["application/vnd.svd"]),
	("svg", &["image/svg+xml"]),
	("svgz", &["image/svg+xml"]),
	("swa", &["application/x-director"]),
	("swf", &["application/vnd.adobe.flash.movie", "application/x-shockwave-flash"]),
	("swi", &["application/vnd.aristanetworks.swi"]),
	("swidtag", &["application/swid+xml"]),
	("sxc", &["application/vnd.sun.xml.calc"]),
	("sxd", &["application/vnd.sun.xml.draw"]),
	("sxg", &["application/vnd.sun.xml.writer.global"]),
	("sxi", &["application/vnd.sun.xml.impress"]),
	("sxm", &["application/vnd.sun.xml.math"]),
	("sxw", &["application/vnd.sun.xml.writer"]),
	("t", &["text/troff"]),
	("t3", &["application/x-t3vm-image"]),
	("t38", &["image/t38"]),
	("taglet", &["application/vnd.mynfc"]),
	("tao", &["application/vnd.tao.intent-module-archive"]),
	("tap", &["image/vnd.tencent.tap"]),
	("tar", &["application/x-tar"]),
	("tcap", &["application/vnd.3gpp2.tcap"]),
	("tcl", &["application/x-tcl"]),
	("td", &["application/urc-targetdesc+xml"]),
	("teacher", &["application/vnd.smart.teacher"]),
	("tei", &["application/tei+xml"]),
	("teicorpus", &["application/tei+xml"]),
	("testrunconfig", &["application/xml"]),
	("testsettings", &["application/xml"]),
	("tex", &["text/x-tex", "application/x-tex"]),
	("texi", &["application/x-texinfo"]),
	("texinfo", &["application/x-texinfo"]),
	("text", &["text/plain"]),
	("tfi", &["application/thraud+xml"]),
	("tfm", &["application/x-tex-tfm"]),
	("tfx", &["image/tiff-fx"]),
	("tga", &["image/x-tga"]),
	("tgz", &["application/x-compressed"]),
	("thmx", &["application/vnd.ms-officetheme"]),
	("thn", &["application/octet-stream"]),
	("tif", &["image/tiff"]),
	("tiff", &["image/tiff"]),
	("tk", &["application/x-tcl"]),
	("tlh", &["text/plain"]),
	("tli", &["text/plain"]),
	("tmo", &["application/vnd.tmobile-livetv"]),
	("toc", &["application/octet-stream"]),
	("toml", &["application/toml", "text/toml", "text/x-toml"]),
	("torrent", &["application/x-bittorrent"]),
	("tpl", &["application/vnd.groove-tool-template"]),
	("tpt", &["application/vnd.trid.tpt"]),
	("tr", &["text/troff", "application/x-troff"]),
	("tra", &["application/vnd.trueapp"]),
	("trig", &["application/trig"]),
	("trm", &["application/x-msterminal"]),
	("trx", &["application/xml"]),
	("ts", &["video/mp2t", "video/vnd.dlna.mpeg-tts"]),
	("tsd", &["application/timestamped-data"]),
	("tsv", &["text/tab-separated-values"]),
	("ttc", &["font/collection"]),
	("ttf", &["font/ttf", "application/x-font-ttf", "application/font-sfnt"]),
	("ttl", &["text/turtle"]),
	("ttml", &["application/ttml+xml"]),
	("tts", &["video/vnd.dlna.mpeg-tts"]),
	("twd", &["application/vnd.simtech-mindmapper"]),
	("twds", &["application/vnd.simtech-mindmapper"]),
	("txd", &["application/vnd.genomatix.tuxedo"]),
	("txf", &["application/vnd.mobius.txf"]),
	("txt", &["text/plain"]),
	("u32", &["application/x-authorware-bin"]),
	("u8dsn", &["message/global-delivery-status"]),
	("u8hdr", &["message/global-headers"]),
	("u8mdn", &["message/global-disposition-notification"]),
	("u8msg", &["message/global"]),
	("ubj", &["application/ubjson"]),
	("udeb", &["application/x-debian-package"]),
	("ufd", &["application/vnd.ufdl"]),
	("ufdl", &["application/vnd.ufdl"]),
	("uls", &["text/iuls"]),
	("ulx", &["application/x-glulx"]),
	("umj", &["application/vnd.umajin"]),
	("unityweb", &["application/vnd.unity"]),
	("uoml", &["application/vnd.uoml+xml"]),
	("uri", &["text/uri-list"]),
	("uris", &["text/uri-list"]),
	("urls", &["text/uri-list"]),
	("usdz", &["model/vnd.usdz+zip"]),
	("user", &["text/plain"]),
	("ustar", &["application/x-ustar"]),
	("utz", &["application/vnd.uiq.theme"]),
	("uu", &["text/x-uuencode"]),
	("uva", &["audio/vnd.dece.audio"]),
	("uvd", &["application/vnd.dece.data"]),
	("uvf", &["application/vnd.dece.data"]),
	("uvg", &["image/vnd.dece.graphic"]),
	("uvh", &["video/vnd.dece.hd"]),
	("uvi", &["image/vnd.dece.graphic"]),
	("uvm", &["video/vnd.dece.mobile"]),
	("uvp", &["video/vnd.dece.pd"]),
	("uvs", &["video/vnd.dece.sd"]),
	("uvt", &["application/vnd.dece.ttml+xml"]),
	("uvu", &["video/vnd.uvvu.mp4"]),
	("uvv", &["video/vnd.dece.video"]),
	("uvva", &["audio/vnd.dece.audio"]),
	("uvvd", &["application/vnd.dece.data"]),
	("uvvf", &["application/vnd.dece.data"]),
	("uvvg", &["image/vnd.dece.graphic"]),
	("uvvh", &["video/vnd.dece.hd"]),
	("uvvi", &["image/vnd.dece.graphic"]),
	("uvvm", &["video/vnd.dece.mobile"]),
	("uvvp", &["video/vnd.dece.pd"]),
	("uvvs", &["video/vnd.dece.sd"]),
	("uvvt", &["application/vnd.dece.ttml+xml"]),
	("uvvu", &["video/vnd.uvvu.mp4"]),
	("uvvv", &["video/vnd.dece.video"]),
	("uvvx", &["application/vnd.dece.unspecified"]),
	("uvvz", &["application/vnd.dece.zip"]),
	("uvx", &["application/vnd.dece.unspecified"]),
	("uvz", &["application/vnd.dece.zip"]),
	("vb", &["text/plain"]),
	("vbdproj", &["text/plain"]),
	("vbk", &["video/mpeg"]),
	("vbox", &["application/x-virtualbox-vbox"]),
	("vbox-extpack", &["application/x-virtualbox-vbox-extpack"]),
	("vbproj", &["text/plain"]),
	("vbs", &["text/vbscript"]),
	("vcard", &["text/vcard"]),
	("vcd", &["application/x-cdlink"]),
	("vcf", &["text/x-vcard"]),
	("vcg", &["application/vnd.groove-vcard"]),
	("vcproj", &["application/xml"]),
	("vcs", &["text/x-vcalendar"]),
	("vcx", &["application/vnd.vcx"]),
	("vcxproj", &["application/xml"]),
	("vddproj", &["text/plain"]),
	("vdi", &["application/x-virtualbox-vdi"]),
	("vdp", &["text/plain"]),
	("vdproj", &["text/plain"]),
	("vds", &["model/vnd.sap.vds"]),
	("vdx", &["application/vnd.ms-visio.viewer"]),
	("vhd", &["application/x-virtualbox-vhd"]),
	("vis", &["application/vnd.visionary"]),
	("viv", &["video/vnd.vivo"]),
	("vmdk", &["application/x-virtualbox-vmdk"]),
	("vml", &["text/xml"]),
	("vob", &["video/x-ms-vob"]),
	("vor", &["application/vnd.stardivision.writer"]),
	("vox", &["application/x-authorware-bin"]),
	("vrml", &["model/vrml"]),
	("vscontent", &["application/xml"]),
	("vsct", &["text/xml"]),
	("vsd", &["application/vnd.visio"]),
	("vsf", &["application/vnd.vsf"]),
	("vsi", &["application/ms-vsi"]),
	("vsix", &["application/vsix"]),
	("vsixlangpack", &["text/xml"]),
	("vsixmanifest", &["text/xml"]),
	("vsmdi", &["application/xml"]),
	("vspscc", &["text/plain"]),
	("vss", &["application/vnd.visio"]),
	("vsscc", &["text/plain"]),
	("vssettings", &["text/xml"]),
	("vssscc", &["text/plain"]),
	("vst", &["application/vnd.visio"]),
	("vstemplate", &["text/xml"]),
	("vsto", &["application/x-ms-vsto"]),
	("vsw", &["application/vnd.visio"]),
	("vsx", &["application/vnd.visio"]),
	("vtf", &["image/vnd.valve.source.texture"]),
	("vtt", &["text/vtt"]),
	("vtu", &["model/vnd.vtu"]),
	("vtx", &["application/vnd.visio"]),
	("vxml", &["application/voicexml+xml"]),
	("w3d", &["application/x-director"]),
	("wad", &["application/x-doom"]),
	("wadl", &["application/vnd.sun.wadl+xml"]),
	("war", &["application/java-archive"]),
	("wasm", &["application/wasm"]),
	("wav", &["audio/wav", "audio/w-wav"]),
	("wave", &["audio/wav", "audio/w-wav"]),
	("wax", &["audio/x-ms-wax"]),
	("wbk", &["application/msword"]),
	("wbmp", &["image/vnd.wap.wbmp"]),
	("wbs", &["application/vnd.criticaltools.wbs+xml"]),
	("wbxml", &["application/vnd.wap.wbxml"]),
	("wcm", &["application/vnd.ms-works"]),
	("wdb", &["application/vnd.ms-works"]),
	("wdp", &["image/vnd.ms-photo"]),
	("weba", &["audio/webm"]),
	("webapp", &["application/x-web-app-manifest+json"]),
	("webarchive", &["application/x-safari-webarchive"]),
	("webm", &["video/webm"]),
	("webmanifest", &["application/manifest+json"]),
	("webp", &["image/webp"]),
	("webtest", &["application/xml"]),
	("wg", &["application/vnd.pmi.widget"]),
	("wgt", &["application/widget"]),
	("wiq", &["application/xml"]),
	("wiz", &["application/msword"]),
	("wks", &["application/vnd.ms-works"]),
	("wlmp", &["application/wlmoviemaker"]),
	("wlpginstall", &["application/x-wlpg-detect"]),
	("wlpginstall3", &["application/x-wlpg3-detect"]),
	("wm", &["video/x-ms-wm"]),
	("wma", &["audio/x-ms-wma"]),
	("wmd", &["application/x-ms-wmd"]),
	("wmf", &["image/wmf"]),
	("wml", &["text/vnd.wap.wml"]),
	("wmlc", &["application/vnd.wap.wmlc"]),
	("wmls", &["text/vnd.wap.wmlscript"]),
	("wmlsc", &["application/vnd.wap.wmlscriptc"]),
	("wmp", &["video/x-ms-wmp"]),
	("wmv", &["video/x-ms-wmv"]),
	("wmx", &["video/x-ms-wmx"]),
	("wmz", &["application/x-ms-wmz"]),
	("woff", &["font/woff", "application/font-woff"]),
	("woff2", &["font/woff2"]),
	("wpd", &["application/vnd.wordperfect"]),
	("wpl", &["application/vnd.ms-wpl"]),
	("wps", &["application/vnd.ms-works"]),
	("wqd", &["application/vnd.wqd"]),
	("wri", &["application/x-mswrite"]),
	("wrl", &["model/vrml", "x-world/x-vrml"]),
	("wrz", &["x-world/x-vrml"]),
	("wsc", &["message/vnd.wfa.wsc", "text/scriptlet"]),
	("wsdl", &["application/wsdl+xml", "text/xml"]),
	("wspolicy", &["application/wspolicy+xml"]),
	("wtb", &["application/vnd.webturbo"]),
	("wvx", &["video/x-ms-wvx"]),
	("x", &["application/directx"]),
	("x32", &["application/x-authorware-bin"]),
	("x3d", &["model/x3d+xml"]),
	("x3db", &["model/x3d+binary", "model/x3d+fastinfoset"]),
	("x3dbz", &["model/x3d+binary"]),
	("x3dv", &["model/x3d+vrml"]),
	("x3dvz", &["model/x3d+vrml"]),
	("x3dz", &["model/x3d+xml"]),
	("x_b", &["model/vnd.parasolid.transmit.binary"]),
	("x_t", &["model/vnd.parasolid.transmit.text"]),
	("xaf", &["x-world/x-vrml"]),
	("xaml", &["application/xaml+xml"]),
	("xap", &["application/x-silverlight-app"]),
	("xar", &["application/vnd.xara"]),
	("xav", &["application/xcap-att+xml"]),
	("xbap", &["application/x-ms-xbap"]),
	("xbd", &["application/vnd.fujixerox.docuworks.binder"]),
	("xbm", &["image/x-xbitmap"]),
	("xca", &["application/xcap-caps+xml"]),
	("xcf", &["image/x-xcf"]),
	("xcs", &["application/calendar+xml"]),
	("xdf", &["application/xcap-diff+xml"]),
	("xdm", &["application/vnd.syncml.dm+xml"]),
	("xdp", &["application/vnd.adobe.xdp+xml"]),
	("xdr", &["text/plain"]),
	("xdssc", &["application/dssc+xml"]),
	("xdw", &["application/vnd.fujixerox.docuworks"]),
	("xel", &["application/xcap-el+xml"]),
	("xenc", &["application/xenc+xml"]),
	("xer", &["application/patch-ops-error+xml"]),
	("xfdf", &["application/vnd.adobe.xfdf"]),
	("xfdl", &["application/vnd.xfdl"]),
	("xht", &["application/xhtml+xml"]),
	("xhtml", &["application/xhtml+xml"]),
	("xhvml", &["application/xv+xml"]),
	("xif", &["image/vnd.xiff"]),
	("xla", &["application/vnd.ms-excel"]),
	("xlam", &["application/vnd.ms-excel.addin.macroenabled.12"]),
	("xlc", &["application/vnd.ms-excel"]),
	("xld", &["application/vnd.ms-excel"]),
	("xlf", &["application/xliff+xml", "application/x-xliff+xml"]),
	("xlk", &["application/vnd.ms-excel"]),
	("xll", &["application/vnd.ms-excel"]),
	("xlm", &["application/vnd.ms-excel"]),
	("xls", &["application/vnd.ms-excel"]),
	("xlsb", &["application/vnd.ms-excel.sheet.binary.macroenabled.12"]),
	("xlsm", &["application/vnd.ms-excel.sheet.macroenabled.12"]),
	("xlsx", &["application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"]),
	("xlt", &["application/vnd.ms-excel"]),
	("xltm", &["application/vnd.ms-excel.template.macroenabled.12"]),
	("xltx", &["application/vnd.openxmlformats-officedocument.spreadsheetml.template"]),
	("xlw", &["application/vnd.ms-excel"]),
	("xm", &["audio/xm"]),
	("xml", &["application/xml", "text/xml"]),
	("xmp", &["application/octet-stream"]),
	("xmta", &["application/xml"]),
	("xns", &["application/xcap-ns+xml"]),
	("xo", &["application/vnd.olpc-sugar"]),
	("xof", &["x-world/x-vrml"]),
	("xoml", &["text/plain"]),
	("xop", &["application/xop+xml"]),
	("xpi", &["application/x-xpinstall"]),
	("xpl", &["application/xproc+xml"]),
	("xpm", &["image/x-xpixmap"]),
	("xpr", &["application/vnd.is-xpr"]),
	("xps", &["application/vnd.ms-xpsdocument"]),
	("xpw", &["application/vnd.intercon.formnet"]),
	("xpx", &["application/vnd.intercon.formnet"]),
	("xrm-ms", &["text/xml"]),
	("xsc", &["application/xml"]),
	("xsd", &["application/xml", "text/xml"]),
	("xsf", &["application/xml", "text/xml"]),
	("xsl", &["application/xslt+xml", "text/xml"]),
	("xslt", &["application/xslt+xml", "text/xml"]),
	("xsm", &["application/vnd.syncml+xml"]),
	("xsn", &["application/octet-stream"]),
	("xspf", &["application/xspf+xml"]),
	("xss", &["application/xml"]),
	("xtp", &["application/octet-stream"]),
	("xul", &["application/vnd.mozilla.xul+xml"]),
	("xvm", &["application/xv+xml"]),
	("xvml", &["application/xv+xml"]),
	("xwd", &["image/x-xwindowdump"]),
	("xyz", &["chemical/x-xyz"]),
	("xz", &["application/x-xz"]),
	("yaml", &["text/yaml", "text/x-yaml"]),
	("yang", &["application/yang"]),
	("yin", &["application/yin+xml"]),
	("yml", &["text/yaml", "text/x-yaml"]),
	("ymp", &["text/x-suse-ymp"]),
	("z", &["application/x-compress"]),
	("z1", &["application/x-zmachine"]),
	("z2", &["application/x-zmachine"]),
	("z3", &["application/x-zmachine"]),
	("z4", &["application/x-zmachine"]),
	("z5", &["application/x-zmachine"]),
	("z6", &["application/x-zmachine"]),
	("z7", &["application/x-zmachine"]),
	("z8", &["application/x-zmachine"]),
	("zaz", &["application/vnd.zzazz.deck+xml"]),
	("zip", &["application/zip"]),
	("zir", &["application/vnd.zul"]),
	("zirz", &["application/vnd.zul"]),
	("zmm", &["application/vnd.handheld-entertainment+xml"]),
	("zsh", &["application/x-shellscript", "application/x-sh", "text/x-shellscript"]),
	("zst", &["application/zstd"]),
];
