#[cfg(feature = "debugger")]
use super::disassembler::AddressingMode;
use num_enum::TryFromPrimitive;

#[cfg_attr(feature = "debugger", derive(Debug))]
#[derive(TryFromPrimitive, Clone, Copy)]
#[repr(u8)]
pub enum Opcode {
    Brk = 0x00,
    OraIndX = 0x01,
    OraZp = 0x05,
    AslZp = 0x06,
    Php = 0x08,
    OraImm = 0x09,
    AslAcc = 0x0a,
    OraAbs = 0x0d,
    AslAbs = 0x0e,

    Bpl = 0x10,
    OraIndY = 0x11,
    OraZpX = 0x15,
    AslZpX = 0x16,
    Clc = 0x18,
    OraAbsY = 0x19,
    OraAbsX = 0x1d,
    AslAbsX = 0x1e,

    JsrAbs = 0x20,
    AndIndX = 0x21,
    BitZp = 0x24,
    AndZp = 0x25,
    RolZp = 0x26,
    Plp = 0x28,
    AndImm = 0x29,
    RolAcc = 0x2a,
    BitAbs = 0x2c,
    AndAbs = 0x2d,
    RolAbs = 0x2e,

    Bmi = 0x30,
    AndIndY = 0x31,
    AndZpX = 0x35,
    RolZpX = 0x36,
    Sec = 0x38,
    AndAbsY = 0x39,
    AndAbsX = 0x3d,
    RolAbsX = 0x3e,

    Rti = 0x40,
    EorIndX = 0x41,
    EorZp = 0x45,
    LsrZp = 0x46,
    Pha = 0x48,
    EorImm = 0x49,
    LsrAcc = 0x4a,
    JmpAbs = 0x4c,
    EorAbs = 0x4d,
    LsrAbs = 0x4e,

    Bvc = 0x50,
    EorIndY = 0x51,
    EorZpX = 0x55,
    LsrZpX = 0x56,
    Cli = 0x58,
    EorAbsY = 0x59,
    EorAbsX = 0x5d,
    LsrAbsX = 0x5e,

    Rts = 0x60,
    AdcIndX = 0x61,
    AdcZp = 0x65,
    RorZp = 0x66,
    Pla = 0x68,
    AdcImm = 0x69,
    RorAcc = 0x6a,
    JmpInd = 0x6c,
    AdcAbs = 0x6d,
    RorAbs = 0x6e,

    Bvs = 0x70,
    AdcIndY = 0x71,
    AdcZpX = 0x75,
    RorZpX = 0x76,
    Sei = 0x78,
    AdcAbsY = 0x79,
    AdcAbsX = 0x7d,
    RorAbsX = 0x7e,

    StaIndX = 0x81,
    StyZp = 0x84,
    StaZp = 0x85,
    StxZp = 0x86,
    Dey = 0x88,
    Txa = 0x8a,
    StyAbs = 0x8c,
    StaAbs = 0x8d,
    StxAbs = 0x8e,

    Bcc = 0x90,
    StaIndY = 0x91,
    StyZpX = 0x94,
    StaZpX = 0x95,
    StxZpY = 0x96,
    Tya = 0x98,
    StaAbsY = 0x99,
    Txs = 0x9a,
    StaAbsX = 0x9d,

    LdyImm = 0xa0,
    LdaIndX = 0xa1,
    LdxImm = 0xa2,
    LdyZp = 0xa4,
    LdaZp = 0xa5,
    LdxZp = 0xa6,
    Tay = 0xa8,
    LdaImm = 0xa9,
    Tax = 0xaa,
    LdyAbs = 0xac,
    LdaAbs = 0xad,
    LdxAbs = 0xae,

    Bcs = 0xb0,
    LdaIndY = 0xb1,
    LdyZpX = 0xb4,
    LdaZpX = 0xb5,
    LdxZpY = 0xb6,
    Clv = 0xb8,
    LdaAbsY = 0xb9,
    Tsx = 0xba,
    LdyAbsX = 0xbc,
    LdaAbsX = 0xbd,
    LdxAbsY = 0xbe,

    CpyImm = 0xc0,
    CmpIndX = 0xc1,
    CpyZp = 0xc4,
    CmpZp = 0xc5,
    DecZp = 0xc6,
    Iny = 0xc8,
    CmpImm = 0xc9,
    Dex = 0xca,
    CpyAbs = 0xcc,
    CmpAbs = 0xcd,
    DecAbs = 0xce,

    Bne = 0xd0,
    CmpIndY = 0xd1,
    CmpZpX = 0xd5,
    DecZpX = 0xd6,
    Cld = 0xd8,
    CmpAbsY = 0xd9,
    CmpAbsX = 0xdd,
    DecAbsX = 0xde,

    CpxImm = 0xe0,
    SbcIndX = 0xe1,
    CpxZp = 0xe4,
    SbcZp = 0xe5,
    IncZp = 0xe6,
    Inx = 0xe8,
    SbcImm = 0xe9,
    Nop = 0xea,
    CpxAbs = 0xec,
    SbcAbs = 0xed,
    IncAbs = 0xee,

    Beq = 0xf0,
    SbcIndY = 0xf1,
    SbcZpX = 0xf5,
    IncZpX = 0xf6,
    Sed = 0xf8,
    SbcAbsY = 0xf9,
    SbcAbsX = 0xfd,
    IncAbsX = 0xfe,
}

impl Opcode {
    pub fn cycles(&self) -> u8 {
        match self {
            Opcode::Brk => 7,
            Opcode::OraIndX => 6,
            Opcode::OraZp => 3,
            Opcode::AslZp => 5,
            Opcode::Php => 3,
            Opcode::OraImm => 2,
            Opcode::AslAcc => 2,
            Opcode::OraAbs => 4,
            Opcode::AslAbs => 6,

            Opcode::Bpl => 2,
            Opcode::OraIndY => 5,
            Opcode::OraZpX => 4,
            Opcode::AslZpX => 6,
            Opcode::Clc => 2,
            Opcode::OraAbsY => 4,
            Opcode::OraAbsX => 4,
            Opcode::AslAbsX => 7,

            Opcode::JsrAbs => 6,
            Opcode::AndIndX => 6,
            Opcode::BitZp => 3,
            Opcode::AndZp => 3,
            Opcode::RolZp => 5,
            Opcode::Plp => 4,
            Opcode::AndImm => 2,
            Opcode::RolAcc => 2,
            Opcode::BitAbs => 4,
            Opcode::AndAbs => 4,
            Opcode::RolAbs => 6,

            Opcode::Bmi => 2,
            Opcode::AndIndY => 5,
            Opcode::AndZpX => 4,
            Opcode::RolZpX => 6,
            Opcode::Sec => 2,
            Opcode::AndAbsY => 4,
            Opcode::AndAbsX => 4,
            Opcode::RolAbsX => 7,

            Opcode::Rti => 6,
            Opcode::EorIndX => 6,
            Opcode::EorZp => 3,
            Opcode::LsrZp => 5,
            Opcode::Pha => 3,
            Opcode::EorImm => 2,
            Opcode::LsrAcc => 2,
            Opcode::JmpAbs => 3,
            Opcode::EorAbs => 4,
            Opcode::LsrAbs => 6,

            Opcode::Bvc => 2,
            Opcode::EorIndY => 5,
            Opcode::EorZpX => 4,
            Opcode::LsrZpX => 6,
            Opcode::Cli => 2,
            Opcode::EorAbsY => 4,
            Opcode::EorAbsX => 4,
            Opcode::LsrAbsX => 7,

            Opcode::Rts => 6,
            Opcode::AdcIndX => 6,
            Opcode::AdcZp => 3,
            Opcode::RorZp => 5,
            Opcode::Pla => 4,
            Opcode::AdcImm => 2,
            Opcode::RorAcc => 2,
            Opcode::JmpInd => 5,
            Opcode::AdcAbs => 4,
            Opcode::RorAbs => 6,

            Opcode::Bvs => 2,
            Opcode::AdcIndY => 5,
            Opcode::AdcZpX => 4,
            Opcode::RorZpX => 6,
            Opcode::Sei => 2,
            Opcode::AdcAbsY => 4,
            Opcode::AdcAbsX => 4,
            Opcode::RorAbsX => 7,

            Opcode::StaIndX => 6,
            Opcode::StyZp => 3,
            Opcode::StaZp => 3,
            Opcode::StxZp => 3,
            Opcode::Dey => 2,
            Opcode::Txa => 2,
            Opcode::StyAbs => 4,
            Opcode::StaAbs => 4,
            Opcode::StxAbs => 4,

            Opcode::Bcc => 2,
            Opcode::StaIndY => 6,
            Opcode::StyZpX => 4,
            Opcode::StaZpX => 4,
            Opcode::StxZpY => 4,
            Opcode::Tya => 2,
            Opcode::StaAbsY => 5,
            Opcode::Txs => 2,
            Opcode::StaAbsX => 5,

            Opcode::LdyImm => 2,
            Opcode::LdaIndX => 6,
            Opcode::LdxImm => 2,
            Opcode::LdyZp => 3,
            Opcode::LdaZp => 3,
            Opcode::LdxZp => 3,
            Opcode::Tay => 2,
            Opcode::LdaImm => 2,
            Opcode::Tax => 2,
            Opcode::LdyAbs => 4,
            Opcode::LdaAbs => 4,
            Opcode::LdxAbs => 4,

            Opcode::Bcs => 2,
            Opcode::LdaIndY => 5,
            Opcode::LdyZpX => 4,
            Opcode::LdaZpX => 4,
            Opcode::LdxZpY => 4,
            Opcode::Clv => 2,
            Opcode::LdaAbsY => 4,
            Opcode::Tsx => 2,
            Opcode::LdyAbsX => 4,
            Opcode::LdaAbsX => 4,
            Opcode::LdxAbsY => 4,

            Opcode::CpyImm => 2,
            Opcode::CmpIndX => 6,
            Opcode::CpyZp => 3,
            Opcode::CmpZp => 3,
            Opcode::DecZp => 5,
            Opcode::Iny => 2,
            Opcode::CmpImm => 2,
            Opcode::Dex => 2,
            Opcode::CpyAbs => 4,
            Opcode::CmpAbs => 4,
            Opcode::DecAbs => 6,

            Opcode::Bne => 2,
            Opcode::CmpIndY => 5,
            Opcode::CmpZpX => 4,
            Opcode::DecZpX => 6,
            Opcode::Cld => 2,
            Opcode::CmpAbsY => 4,
            Opcode::CmpAbsX => 4,
            Opcode::DecAbsX => 7,

            Opcode::CpxImm => 2,
            Opcode::SbcIndX => 6,
            Opcode::CpxZp => 3,
            Opcode::SbcZp => 3,
            Opcode::IncZp => 5,
            Opcode::Inx => 2,
            Opcode::SbcImm => 2,
            Opcode::Nop => 2,
            Opcode::CpxAbs => 4,
            Opcode::SbcAbs => 4,
            Opcode::IncAbs => 6,

            Opcode::Beq => 2,
            Opcode::SbcIndY => 5,
            Opcode::SbcZpX => 4,
            Opcode::IncZpX => 6,
            Opcode::Sed => 2,
            Opcode::SbcAbsY => 4,
            Opcode::SbcAbsX => 4,
            Opcode::IncAbsX => 7,
        }
    }

    #[cfg(feature = "debugger")]
    pub fn addressing_mode(&self) -> AddressingMode {
        match self {
            Opcode::Brk => AddressingMode::Implied,
            Opcode::OraIndX => AddressingMode::IndirectX,
            Opcode::OraZp => AddressingMode::ZeroPage,
            Opcode::AslZp => AddressingMode::ZeroPage,
            Opcode::Php => AddressingMode::Implied,
            Opcode::OraImm => AddressingMode::Immediate,
            Opcode::AslAcc => AddressingMode::Accumulator,
            Opcode::OraAbs => AddressingMode::Absolute,
            Opcode::AslAbs => AddressingMode::Absolute,

            Opcode::Bpl => AddressingMode::Relative,
            Opcode::OraIndY => AddressingMode::IndirectY,
            Opcode::OraZpX => AddressingMode::ZeroPageX,
            Opcode::AslZpX => AddressingMode::ZeroPageX,
            Opcode::Clc => AddressingMode::Implied,
            Opcode::OraAbsY => AddressingMode::AbsoluteY,
            Opcode::OraAbsX => AddressingMode::AbsoluteX,
            Opcode::AslAbsX => AddressingMode::AbsoluteX,

            Opcode::JsrAbs => AddressingMode::Absolute,
            Opcode::AndIndX => AddressingMode::IndirectX,
            Opcode::BitZp => AddressingMode::ZeroPage,
            Opcode::AndZp => AddressingMode::ZeroPage,
            Opcode::RolZp => AddressingMode::ZeroPage,
            Opcode::Plp => AddressingMode::Implied,
            Opcode::AndImm => AddressingMode::Immediate,
            Opcode::RolAcc => AddressingMode::Accumulator,
            Opcode::BitAbs => AddressingMode::Absolute,
            Opcode::AndAbs => AddressingMode::Absolute,
            Opcode::RolAbs => AddressingMode::Absolute,

            Opcode::Bmi => AddressingMode::Relative,
            Opcode::AndIndY => AddressingMode::IndirectY,
            Opcode::AndZpX => AddressingMode::ZeroPageX,
            Opcode::RolZpX => AddressingMode::ZeroPageX,
            Opcode::Sec => AddressingMode::Implied,
            Opcode::AndAbsY => AddressingMode::AbsoluteY,
            Opcode::AndAbsX => AddressingMode::AbsoluteX,
            Opcode::RolAbsX => AddressingMode::AbsoluteX,

            Opcode::Rti => AddressingMode::Implied,
            Opcode::EorIndX => AddressingMode::IndirectX,
            Opcode::EorZp => AddressingMode::ZeroPage,
            Opcode::LsrZp => AddressingMode::ZeroPage,
            Opcode::Pha => AddressingMode::Implied,
            Opcode::EorImm => AddressingMode::Immediate,
            Opcode::LsrAcc => AddressingMode::Accumulator,
            Opcode::JmpAbs => AddressingMode::Absolute,
            Opcode::EorAbs => AddressingMode::Absolute,
            Opcode::LsrAbs => AddressingMode::Absolute,

            Opcode::Bvc => AddressingMode::Relative,
            Opcode::EorIndY => AddressingMode::IndirectY,
            Opcode::EorZpX => AddressingMode::ZeroPageX,
            Opcode::LsrZpX => AddressingMode::ZeroPageX,
            Opcode::Cli => AddressingMode::Implied,
            Opcode::EorAbsY => AddressingMode::AbsoluteY,
            Opcode::EorAbsX => AddressingMode::AbsoluteX,
            Opcode::LsrAbsX => AddressingMode::AbsoluteX,

            Opcode::Rts => AddressingMode::Implied,
            Opcode::AdcIndX => AddressingMode::IndirectX,
            Opcode::AdcZp => AddressingMode::ZeroPage,
            Opcode::RorZp => AddressingMode::ZeroPage,
            Opcode::Pla => AddressingMode::Implied,
            Opcode::AdcImm => AddressingMode::Immediate,
            Opcode::RorAcc => AddressingMode::Accumulator,
            Opcode::JmpInd => AddressingMode::Indirect,
            Opcode::AdcAbs => AddressingMode::Absolute,
            Opcode::RorAbs => AddressingMode::Absolute,

            Opcode::Bvs => AddressingMode::Relative,
            Opcode::AdcIndY => AddressingMode::IndirectY,
            Opcode::AdcZpX => AddressingMode::ZeroPageX,
            Opcode::RorZpX => AddressingMode::ZeroPageX,
            Opcode::Sei => AddressingMode::Implied,
            Opcode::AdcAbsY => AddressingMode::AbsoluteY,
            Opcode::AdcAbsX => AddressingMode::AbsoluteX,
            Opcode::RorAbsX => AddressingMode::AbsoluteX,

            Opcode::StaIndX => AddressingMode::IndirectX,
            Opcode::StyZp => AddressingMode::ZeroPage,
            Opcode::StaZp => AddressingMode::ZeroPage,
            Opcode::StxZp => AddressingMode::ZeroPage,
            Opcode::Dey => AddressingMode::Implied,
            Opcode::Txa => AddressingMode::Implied,
            Opcode::StyAbs => AddressingMode::Absolute,
            Opcode::StaAbs => AddressingMode::Absolute,
            Opcode::StxAbs => AddressingMode::Absolute,

            Opcode::Bcc => AddressingMode::Relative,
            Opcode::StaIndY => AddressingMode::IndirectY,
            Opcode::StyZpX => AddressingMode::ZeroPageX,
            Opcode::StaZpX => AddressingMode::ZeroPageX,
            Opcode::StxZpY => AddressingMode::ZeroPageY,
            Opcode::Tya => AddressingMode::Implied,
            Opcode::StaAbsY => AddressingMode::AbsoluteY,
            Opcode::Txs => AddressingMode::Implied,
            Opcode::StaAbsX => AddressingMode::AbsoluteX,

            Opcode::LdyImm => AddressingMode::Immediate,
            Opcode::LdaIndX => AddressingMode::IndirectX,
            Opcode::LdxImm => AddressingMode::Immediate,
            Opcode::LdyZp => AddressingMode::ZeroPage,
            Opcode::LdaZp => AddressingMode::ZeroPage,
            Opcode::LdxZp => AddressingMode::ZeroPage,
            Opcode::Tay => AddressingMode::Implied,
            Opcode::LdaImm => AddressingMode::Immediate,
            Opcode::Tax => AddressingMode::Implied,
            Opcode::LdyAbs => AddressingMode::Absolute,
            Opcode::LdaAbs => AddressingMode::Absolute,
            Opcode::LdxAbs => AddressingMode::Absolute,

            Opcode::Bcs => AddressingMode::Relative,
            Opcode::LdaIndY => AddressingMode::IndirectY,
            Opcode::LdyZpX => AddressingMode::ZeroPageX,
            Opcode::LdaZpX => AddressingMode::ZeroPageX,
            Opcode::LdxZpY => AddressingMode::ZeroPageY,
            Opcode::Clv => AddressingMode::Implied,
            Opcode::LdaAbsY => AddressingMode::AbsoluteY,
            Opcode::Tsx => AddressingMode::Implied,
            Opcode::LdyAbsX => AddressingMode::AbsoluteX,
            Opcode::LdaAbsX => AddressingMode::AbsoluteX,
            Opcode::LdxAbsY => AddressingMode::AbsoluteY,

            Opcode::CpyImm => AddressingMode::Immediate,
            Opcode::CmpIndX => AddressingMode::IndirectX,
            Opcode::CpyZp => AddressingMode::ZeroPage,
            Opcode::CmpZp => AddressingMode::ZeroPage,
            Opcode::DecZp => AddressingMode::ZeroPage,
            Opcode::Iny => AddressingMode::Implied,
            Opcode::CmpImm => AddressingMode::Immediate,
            Opcode::Dex => AddressingMode::Implied,
            Opcode::CpyAbs => AddressingMode::Absolute,
            Opcode::CmpAbs => AddressingMode::Absolute,
            Opcode::DecAbs => AddressingMode::Absolute,

            Opcode::Bne => AddressingMode::Relative,
            Opcode::CmpIndY => AddressingMode::IndirectY,
            Opcode::CmpZpX => AddressingMode::ZeroPageX,
            Opcode::DecZpX => AddressingMode::ZeroPageX,
            Opcode::Cld => AddressingMode::Implied,
            Opcode::CmpAbsY => AddressingMode::AbsoluteY,
            Opcode::CmpAbsX => AddressingMode::AbsoluteX,
            Opcode::DecAbsX => AddressingMode::AbsoluteX,

            Opcode::CpxImm => AddressingMode::Immediate,
            Opcode::SbcIndX => AddressingMode::IndirectX,
            Opcode::CpxZp => AddressingMode::ZeroPage,
            Opcode::SbcZp => AddressingMode::ZeroPage,
            Opcode::IncZp => AddressingMode::ZeroPage,
            Opcode::Inx => AddressingMode::Implied,
            Opcode::SbcImm => AddressingMode::Immediate,
            Opcode::Nop => AddressingMode::Implied,
            Opcode::CpxAbs => AddressingMode::Absolute,
            Opcode::SbcAbs => AddressingMode::Absolute,
            Opcode::IncAbs => AddressingMode::Absolute,

            Opcode::Beq => AddressingMode::Relative,
            Opcode::SbcIndY => AddressingMode::IndirectY,
            Opcode::SbcZpX => AddressingMode::ZeroPageX,
            Opcode::IncZpX => AddressingMode::ZeroPageX,
            Opcode::Sed => AddressingMode::Implied,
            Opcode::SbcAbsY => AddressingMode::AbsoluteY,
            Opcode::SbcAbsX => AddressingMode::AbsoluteX,
            Opcode::IncAbsX => AddressingMode::AbsoluteX,
        }
    }
}
