use crate::*;
use serde::{Serialize, Deserialize};
use serde::de::DeserializeOwned;
use std::ops::Add;
use std::fmt::Display;
use std::process::Command;

use near_sdk::AccountId;
use std::str::FromStr;
use std::path::{PathBuf, Path};


impl<T> ToJson for T where T:Serialize+DeserializeOwned {
    fn to_json(&self) -> String {
        serde_json::to_string(&self).unwrap()
    }
}

// impl<T> ToJson for T where T:Serialize+DeserializeOwned+for<'a> Deserialize<'a> {
//     fn to_json(&self) -> String {
//         serde_json::to_string(&self).unwrap()
//     }
// }

impl NearValue for u128 {
    fn to_near_u128(self) -> String {
        (self * 10u128.pow(24)).to_string()
    }
    fn to_ynear_u128(self) -> String {
        (self / 10u128.pow(24)).to_string()
    }
}

impl NearValue for f64 {
    fn to_near_u128(self) -> String {
        (self * 10f64.powf(24.0)).to_string()
    }
    fn to_ynear_u128(self) -> String {
        (&(self / 10f64.powf(24.0)).to_string()[0..26]).to_string()
    }
}

impl Yocto for f64 {
    fn yocto(self) -> String {
        (&(self / 10f64.powf(24.0)).to_string()[0..26]).to_string()
    }
}

impl ContractMethod for NftApprove {
    fn contract_method(&self) -> String {
        "nft_approve".into()
    }
}

impl ContractMethod for PathBuf {
    fn contract_method(&self) -> String {
        "deploy".into()
    }
}

impl ContractMethod for Path {
    fn contract_method(&self) -> String {
        "deploy".into()
    }
}

impl ContractMethod for DeployBuild<'_> {
    fn contract_method(&self) -> String {
        "deploy".into()
    }
}

impl DeployBuild<'_> {
    pub fn build(&self)->String{
        let mut s = format!("--wasmFile {} --accountId {}",self.wasm_file,self.account_id);
        if let Some(v) = self.init_fn {
            s = format!("{} --initFunction '{}'",s,v);
        }
        if let Some(v) = self.init_args {
            s = format!("{} --initArgs '{}'",s,v);
        }
        if let Some(v) = self.key_path {
            s = format!("{} --keyPath {}",s,v);
        }
        s
    }
}

impl<T> NearCliBuilder for T where T:ContractMethod+Serialize {
    fn call_builder(&self, b:CallBuild) -> String {
        let json = serde_json::to_string(&self).unwrap();
        let mut s = format!("{} {} '{}' {} --accountId {}",b.contract,self.contract_method(),json, b.arg, b.account_id);
        if let Some(v) = b.amount {
            s = format!("{} --amount {}",s,v);
        }
        if let Some(v) = b.gas {
            s = format!("{} --gas {}",s,v);
        }
        s
    }

    fn deploy_builder(&self, b: DeployBuild) -> String {
        let json = serde_json::to_string(&self).unwrap();
        let mut s = format!("--wasmFile {} --accountId {}",json,b.account_id);
        if let Some(v) = b.init_fn {
            s = format!("{} --initFunction '{}'",s,v);
        }
        if let Some(v) = b.init_args {
            s = format!("{} --initArgs '{}'",s,v);
        }
        s
    }

    fn create_account_builder(&self, b: CreateAccountBuild) -> String {
        let json = serde_json::to_string(&self).unwrap();
        let mut s = format!("--accountId {} --masterAccount {} --initialBalance {} --keyPath {}",b.account_id,b.master_account,b.initial_balance,b.key_path);
        s
    }
}

impl SubCredentials<'_> {
    pub fn create_sub_credentials(&self){
        let near_cred_path = near_credentials_dir_testnet();
        let root_account_path = PathBuf::from(near_cred_path.as_str()).join(format!("{}.json",self.root_account));
        let root_account_json = std::fs::read_to_string(root_account_path).unwrap();
        self.sub_accounts.iter().for_each(|x|{
            let new_account = format!("{}.{}",x,self.root_account);
            let new_cred = root_account_json
                .replace(
                    self.root_account,
                    new_account.as_str()
                );
            let new_account_path = PathBuf::from_str(near_cred_path.as_str())
                .unwrap()
                .join(format!("{}.json",new_account).as_str());
            println!("writing {:?}",new_account_path);
            std::fs::write(new_account_path,new_cred).unwrap();
        });
    }
}



impl Print for String {
    fn print(&self) {
        println!("{}",self)
    }
}

// TODO - change this to enum
impl NearCommand for String {
    fn near_call(&self) {
        near_command("call",self)
    }

    fn near_view(&self) {
        near_command("view",self)
    }

    fn near_deploy(&self) {
        near_command("deploy",self)
    }

    fn near_create_account(&self) {
        near_command("create-account",self)

    }
}
