//! Functions to transpile Rust to TypeScript.

use crate::{contract::Contract, write_docs, NearImpl, NearMethod, NearSerde};
use std::{
    io::{self, Write},
    ops::Deref,
};
use syn::{
    Attribute, Fields, ImplItemMethod, Item, ItemEnum, ItemImpl, ItemStruct, PathArguments,
    ReturnType, Type,
};

/// Exports common NEAR Rust SDK types based on
/// https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/.
/// Moreover, it adds a header indicating the time and binary that
/// generated these bindings.
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_prelude(&mut buf, " 2021".to_string(), "bin");
/// assert_eq!(String::from_utf8_lossy(&buf), format!(
/// r#"// TypeScript bindings automatically generated by bin v{} {} 2021, DO NOT MODIFY!
///
/// // Exports common NEAR Rust SDK types based on https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/.
///
/// /**
///  * Represents an 64 bits unsigned integer encoded as a `string`.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/json_types/struct.U64.html.
///  */
/// export type U64 = string;
///
/// /**
///  * Represents an 64 bits signed integer encoded as a `string`.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/json_types/struct.I64.html.
///  */
/// export type I64 = string;
///
/// /**
///  * Represents an 128 bits unsigned integer encoded as a `string`.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/json_types/struct.U128.html.
///  */
/// export type U128 = string;
///
/// /**
///  * Represents an 128 bits signed integer encoded as a `string`.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/json_types/struct.I128.html.
///  */
/// export type I128 = string;
///
/// /**
///  * Represents an encoded array of bytes into a `string`.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/json_types/struct.Base64VecU8.html.
///  */
/// export type Base64VecU8 = string;
///
/// /**
///  * Balance is a type for storing amounts of tokens, specified in yoctoNEAR.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/type.Balance.html.
///  */
/// export type Balance = U128;
///
/// /**
///  * Account identifier. This is the human readable UTF8 string which is used internally to index accounts on the network and their respective state.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/struct.AccountId.html.
///  */
/// export type AccountId = string;
///
/// /**
///  * DEPRECATED since 4.0.0.
///  * See https://docs.rs/near-sdk/4.0.0-pre.4/near_sdk/json_types/type.ValidAccountId.html.
///  */
/// export type ValidAccountId = string;
///
/// "#,
///   env!("CARGO_PKG_VERSION"),
///   env!("CARGO_PKG_REPOSITORY"),
///   ));
/// ```
pub fn ts_prelude<W: Write>(buf: &mut W, now: String, bin_name: &str) -> io::Result<()> {
    writeln!(
        buf,
        "// TypeScript bindings automatically generated by {} v{} {}{}, DO NOT MODIFY!\n",
        bin_name,
        env!("CARGO_PKG_VERSION"),
        env!("CARGO_PKG_REPOSITORY"),
        now
    )?;

    let prelude = include_str!("_prelude.ts");
    writeln!(buf, "{}", prelude)?;

    Ok(())
}

/// Emits additional extensions for the main type implemented by the contract.
/// This is used when the contract implements one or more `trait`s.
/// The `name` and `interfaces` fields must be set in order to emit these additional extensions.
///
/// ## Examples
///
/// ```
/// let mut contract = near_syn::contract::Contract::new();
/// contract.name = "Contract".to_string();
/// contract.interfaces.push("NftCore".to_string());
/// contract.interfaces.push("NftEnum".to_string());
/// let mut buf = Vec::new();
/// near_syn::ts::ts_extend_traits(&mut buf, &contract);
/// assert_eq!(String::from_utf8_lossy(&buf), "export interface Contract extends NftCore, NftEnum {}\n\n");
/// ```
pub fn ts_extend_traits<W: Write>(buf: &mut W, contract: &Contract) -> io::Result<()> {
    if !contract.name.is_empty() && !contract.interfaces.is_empty() {
        writeln!(
            buf,
            "export interface {} extends {} {{}}\n",
            contract.name,
            contract.interfaces.join(", ")
        )?;
    }

    Ok(())
}

/// Exports the methods object required by `near-api-js` to be able
/// to find contract methods.
///
/// ### Examples
///
/// ```
/// let mut contract = near_syn::contract::Contract::new();
/// contract.name = "Contract".to_string();
/// contract.view_methods.push("get".to_string());
/// contract.change_methods.push("set".to_string());
/// contract.change_methods.push("insert".to_string());
/// let mut buf = Vec::new();
/// near_syn::ts::ts_contract_methods(&mut buf, &contract);
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"export const ContractMethods = {
///     viewMethods: [
///         "get",
///     ],
///     changeMethods: [
///         "set",
///         "insert",
///     ],
/// };
/// "#);
/// ```
///
/// Both `viewMethods` and `changeMethods` fields must be present in the
/// resulting object, even if either of them are empty.
/// This is required by `near-api-js`.
///
/// ```
/// let mut contract = near_syn::contract::Contract::new();
/// contract.name = "Contract".to_string();
/// contract.view_methods.push("get".to_string());
/// let mut buf = Vec::new();
/// near_syn::ts::ts_contract_methods(&mut buf, &contract);
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"export const ContractMethods = {
///     viewMethods: [
///         "get",
///     ],
///     changeMethods: [
///     ],
/// };
/// "#);
/// ```
pub fn ts_contract_methods<W: Write>(buf: &mut W, contract: &Contract) -> io::Result<()> {
    fn fmt(methods: &Vec<String>) -> String {
        methods
            .iter()
            .map(|m| format!("        {:?},\n", m))
            .collect::<Vec<String>>()
            .join("")
    }

    writeln!(buf, "export const {}Methods = {{", contract.name)?;
    writeln!(
        buf,
        "    viewMethods: [\n{}    ],",
        fmt(&contract.view_methods)
    )?;
    writeln!(
        buf,
        "    changeMethods: [\n{}    ],",
        fmt(&contract.change_methods)
    )?;
    writeln!(buf, "}};")?;

    Ok(())
}

/// Translates a collection of Rust items to TypeScript.
/// It currently translates `type`, `struct`, `enum` and `impl` items to TypeScript.
/// It traverses recursively `mod` definitions with braced content.
/// The inner `mod`' items are flatten into a single TypeScript module.
/// If an item in `items` is not one of the mentioned above, it is ignored.
///
/// Notice how `mod` definitions are flattened:
///
/// ```
/// let mut contract = near_syn::contract::Contract::new();
/// let mut buf = Vec::new();
/// let ast: syn::File = syn::parse2(quote::quote! {
///         /// Doc-comments are translated.
///         type T = u64;
///         mod inner_mod {
///             /// Doc-comments are translated.
///             type S = u64;
///         }
///     }).unwrap();
/// near_syn::ts::ts_items(&mut buf, &ast.items, &mut contract);
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Doc-comments are translated.
///  */
/// export type T = number;
///
/// /**
///  * Doc-comments are translated.
///  */
/// export type S = number;
///
/// "#);
/// ```
///
/// ```
/// let ast: syn::File = syn::parse2(quote::quote! {
///         #[near_bindgen]
///         impl Contract {
///             pub fn get(&self) -> u32 { 42 }
///         }
///
///         #[near_bindgen]
///         impl NftCore for Contract {
///             fn f(&self) -> u32 { 42 }
///         }
///
///         #[near_bindgen]
///         impl NftEnum for Contract {
///             fn g(&self) -> u32 { 42 }
///         }
///
///     }).unwrap();
/// let mut contract = near_syn::contract::Contract::new();
/// contract.forward_traits(&ast.items);
/// let mut buf = Vec::new();
/// near_syn::ts::ts_items(&mut buf, &ast.items, &mut contract);
/// near_syn::ts::ts_extend_traits(&mut buf, &contract);
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  */
/// export interface Contract {
///     /**
///      */
///     get(): Promise<number>;
///
/// }
///
/// /**
///  */
/// export interface NftCore {
///     /**
///      */
///     f(): Promise<number>;
///
/// }
///
/// /**
///  */
/// export interface NftEnum {
///     /**
///      */
///     g(): Promise<number>;
///
/// }
///
/// export interface Contract extends NftCore, NftEnum {}
///
/// "#);
/// ```
pub fn ts_items<W: Write>(buf: &mut W, items: &Vec<Item>, contract: &Contract) -> io::Result<()> {
    for item in items {
        match item {
            Item::Mod(item_mod) => {
                if let Some((_, mod_items)) = &item_mod.content {
                    ts_items(buf, mod_items, contract)?;
                }
            }
            Item::Impl(item_impl) => ts_impl(buf, &item_impl, contract)?,
            Item::Struct(item_struct) => ts_struct(buf, &item_struct)?,
            Item::Enum(item_enum) => ts_enum(buf, &item_enum)?,
            Item::Type(item_type) => ts_typedef(buf, &item_type)?,
            _ => {}
        }
    }

    Ok(())
}

/// Translates an `impl` section to a TypeScript `interface.`
///
/// A `struct` can have multiple `impl` sections with no `trait` to declare additional methods.
/// These `impl`s are emitted with the name of the contract,
/// as TypeScript merges these definitions.
///
/// ```
/// let mut contract = near_syn::contract::Contract::new();
/// let mut buf = Vec::new();
/// near_syn::ts::ts_impl(&mut buf, &syn::parse2(quote::quote! {
///         /// Doc-comments are translated.
///         #[near_bindgen]
///         impl Contract {
///             /// Doc-comments here are translated as well.
///             pub fn get(&self) -> u32 { 42 }
///         }
///     }).unwrap(), &mut contract);
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Doc-comments are translated.
///  */
/// export interface Contract {
///     /**
///      * Doc-comments here are translated as well.
///      */
///     get(): Promise<number>;
///
/// }
///
/// "#);
/// ```
pub fn ts_impl<W: Write>(buf: &mut W, item_impl: &ItemImpl, contract: &Contract) -> io::Result<()> {
    if let Some(methods) = item_impl.bindgen_methods() {
        let mut item_trait = None;
        if let Some(trait_name) = item_impl.get_trait_name() {
            item_trait = contract.traits.get(&trait_name);
            ts_doc(buf, &item_impl.join_attrs(item_trait), "")?;
            writeln!(buf, "export interface {} {{", trait_name)?;
        } else {
            if let Some(impl_name) = item_impl.get_impl_name() {
                ts_doc(buf, &item_impl.attrs, "")?;
                writeln!(buf, "export interface {} {{", impl_name)?;
            } else {
                panic!("name not found")
            }
        }

        for method in methods {
            ts_doc(buf, &method.join_attrs(item_trait), "    ")?;
            writeln!(buf, "    {}\n", ts_sig(&method))?;
        }

        writeln!(buf, "}}\n")?;
    }

    Ok(())
}

/// Generates the corresponding TypeScript bindings for the given `struct`.
/// Doc-comments embedded in the Rust source file are included in the bindings.
/// The `struct` must derive `Serialize` from `serde` in order to
/// generate its corresponding TypeScript bindings.
///
/// ### Examples
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_struct(&mut buf, &syn::parse2(quote::quote! {
///         /// Doc-comments are also translated.
///         #[derive(Serialize)]
///         struct A {
///             /// Doc-comments here are translated as well.
///             field: u32,
///         }
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Doc-comments are also translated.
///  */
/// export type A = {
///     /**
///      * Doc-comments here are translated as well.
///      */
///     field: number;
///
/// }
///
/// "#);
/// ```
///
/// Single-compoenent tuple-structs are converted to TypeScript type synonym.
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_struct(&mut buf, &syn::parse2(quote::quote! {
///         /// Tuple struct with one component.
///         #[derive(Serialize)]
///         struct T(String);
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Tuple struct with one component.
///  */
/// export type T = string;
///
/// "#);
/// ```
///
/// On the other hand,
/// tuple-structs with more than one component,
/// are converted to TypeScript proper tuples.
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_struct(&mut buf, &syn::parse2(quote::quote! {
///         /// Tuple struct with one component.
///         #[derive(Serialize)]
///         struct T(String, u32);
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Tuple struct with one component.
///  */
/// export type T = [string, number];
///
/// "#);
/// ```
///
/// If derive `Serialize` is not found, given `struct` is omitted.
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_struct(&mut buf, &syn::parse2(quote::quote! {
///         struct A { }
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf), "");
/// ```
pub fn ts_struct<W: Write>(buf: &mut W, item_struct: &ItemStruct) -> io::Result<()> {
    if !item_struct.is_serde() {
        return Ok(());
    }

    ts_doc(buf, &item_struct.attrs, "")?;
    match &item_struct.fields {
        Fields::Named(fields) => {
            writeln!(buf, "export type {} = {{", item_struct.ident)?;
            for field in &fields.named {
                let field_name = field.ident.as_ref().unwrap();
                let ty = ts_type(&field.ty);
                ts_doc(buf, &field.attrs, "    ")?;
                writeln!(buf, "    {}: {};\n", field_name, ty)?;
            }
            writeln!(buf, "}}")?;
            writeln!(buf, "")?;
        }
        Fields::Unnamed(fields) => {
            let mut tys = Vec::new();
            for field in &fields.unnamed {
                let ty = ts_type(&field.ty);
                tys.push(ty);
            }
            writeln!(
                buf,
                "export type {} = {};\n",
                item_struct.ident,
                if tys.len() == 1 {
                    tys.get(0).unwrap().clone()
                } else {
                    format!("[{}]", tys.join(", "))
                }
            )?;
        }
        Fields::Unit => panic!("unit struct no supported"),
    }

    Ok(())
}

/// Translates an enum to a TypeScript `enum` or `type` according to the
/// Rust definition.
/// The Rust `enum` must derive `Serialize` from `serde` in order
/// to be translated.
///
/// ### Examples
///
/// For instance, a plain Rust `enum` will be translated to an `enum`.
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_enum(&mut buf, &syn::parse2(quote::quote! {
///         /// Doc-comments are translated.
///         #[derive(Serialize)]
///         enum E {
///             /// Doc-comments here are translated as well.
///             V1,
///         }
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Doc-comments are translated.
///  */
/// export enum E {
///     /**
///      * Doc-comments here are translated as well.
///      */
///     V1,
///
/// }
///
/// "#);
/// ```
pub fn ts_enum<W: Write>(buf: &mut W, item_enum: &ItemEnum) -> io::Result<()> {
    if !item_enum.is_serde() {
        return Ok(());
    }

    ts_doc(buf, &item_enum.attrs, "")?;
    writeln!(buf, "export enum {} {{", item_enum.ident)?;
    for variant in &item_enum.variants {
        ts_doc(buf, &variant.attrs, "    ")?;
        writeln!(buf, "    {},\n", variant.ident)?;
    }
    writeln!(buf, "}}\n")?;

    Ok(())
}

/// Translates a type alias to another type alias in TypeScript.
///
/// ### Examples
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_typedef(&mut buf, &syn::parse2(quote::quote! {
///         /// Doc-comments are translated.
///         type T = u64;
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  * Doc-comments are translated.
///  */
/// export type T = number;
///
/// "#);
/// ```
///
/// If doc-comments are omitted,
/// TypeScript empty doc-comments will be emitted.
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_typedef(&mut buf, &syn::parse2(quote::quote! {
///         type T = u64;
///     }).unwrap());
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"/**
///  */
/// export type T = number;
///
/// "#);
/// ```
pub fn ts_typedef<W: Write>(buf: &mut W, item_type: &syn::ItemType) -> io::Result<()> {
    ts_doc(buf, &item_type.attrs, "")?;
    writeln!(
        buf,
        "export type {} = {};",
        item_type.ident,
        ts_type(&item_type.ty)
    )?;
    writeln!(buf, "")?;

    Ok(())
}

/// Translates `doc` attributes into TypeScript docs.
/// The `indent` argument is used as a prefix for each line emitted.
///
/// ### Examples
///
/// ```
/// let mut buf = Vec::new();
/// near_syn::ts::ts_doc(&mut buf, &syn::parse2::<syn::ItemType>(quote::quote! {
///         /// Doc-comments are translated.
///         type T = u64;
///     }).unwrap().attrs, "    ");
/// assert_eq!(String::from_utf8_lossy(&buf),
/// r#"    /**
///      * Doc-comments are translated.
///      */
/// "#);
/// ```
pub fn ts_doc<W: Write>(buf: &mut W, attrs: &Vec<Attribute>, indent: &str) -> io::Result<()> {
    writeln!(buf, "{}/**", indent)?;
    write_docs(buf, attrs, |l| format!("{} * {}", indent, l.trim_start()))?;
    writeln!(buf, "{} */", indent)?;

    Ok(())
}

/// Return the TypeScript equivalent type of the Rust type represented by `ty`.
///
/// ### Examples
///
/// Rust primitives types and `String` are included.
///
/// ```
/// use syn::parse_str;
/// use near_syn::ts::ts_type;
///
/// assert_eq!(ts_type(&parse_str("bool").unwrap()), "boolean");
/// assert_eq!(ts_type(&parse_str("i8").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("u8").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("i16").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("u16").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("i32").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("u32").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("String").unwrap()), "string");
/// ```
///
/// Rust shared references are supported as well.
///
/// ```
/// # use syn::parse_str;
/// # use near_syn::ts::ts_type;
/// assert_eq!(ts_type(&parse_str("&String").unwrap()), "string");
/// assert_eq!(ts_type(&parse_str("&bool").unwrap()), "boolean");
/// assert_eq!(ts_type(&parse_str("&u32").unwrap()), "number");
/// assert_eq!(ts_type(&parse_str("&TokenId").unwrap()), "TokenId");
/// ```
///
/// Rust standard and collections types, *e.g.*, `Option`, `Vec` and `HashMap`,
/// are included in the translation.
///
/// ```
/// # use syn::parse_str;
/// # use near_syn::ts::ts_type;
/// assert_eq!(ts_type(&parse_str("Option<U64>").unwrap()), "U64|null");
/// assert_eq!(ts_type(&parse_str("Option<String>").unwrap()), "string|null");
/// assert_eq!(ts_type(&parse_str("Vec<ValidAccountId>").unwrap()), "ValidAccountId[]");
/// assert_eq!(ts_type(&parse_str("HashSet<ValidAccountId>").unwrap()), "ValidAccountId[]");
/// assert_eq!(ts_type(&parse_str("BTreeSet<ValidAccountId>").unwrap()), "ValidAccountId[]");
/// assert_eq!(ts_type(&parse_str("HashMap<AccountId, U128>").unwrap()), "Record<AccountId, U128>");
/// assert_eq!(ts_type(&parse_str("BTreeMap<AccountId, U128>").unwrap()), "Record<AccountId, U128>");
/// ```
///
/// Rust nested types are converted to TypeScript as well.
///
/// ```
/// # use syn::parse_str;
/// # use near_syn::ts::ts_type;
/// assert_eq!(ts_type(&parse_str("HashMap<AccountId, Vec<U128>>").unwrap()), "Record<AccountId, U128[]>");
/// assert_eq!(ts_type(&parse_str("Vec<Option<U128>>").unwrap()), "(U128|null)[]");
/// assert_eq!(ts_type(&parse_str("Option<Vec<U128>>").unwrap()), "U128[]|null");
/// assert_eq!(ts_type(&parse_str("Option<Option<U64>>").unwrap()), "U64|null|null");
/// assert_eq!(ts_type(&parse_str("Vec<Vec<U64>>").unwrap()), "U64[][]");
/// assert_eq!(ts_type(&parse_str("(U64)").unwrap()), "U64");
/// assert_eq!(ts_type(&parse_str("(U64, String, Vec<u32>)").unwrap()), "[U64, string, number[]]");
///
/// assert_eq!(ts_type(&parse_str("()").unwrap()), "void");
/// // assert_eq!(ts_type(&parse_str("std::vec::Vec<U64>").unwrap()), "U64[]");
/// ```
///
/// ## Panics
///
/// Panics when standard library generics types are used incorrectly.
/// For example `Option` or `HashMap<U64>`.
/// This situation can only happen on Rust source files that were **not** type-checked by `rustc`.
pub fn ts_type(ty: &Type) -> String {
    #[derive(PartialEq, PartialOrd)]
    enum Assoc {
        Single,
        Vec,
        Or,
    }
    fn single(ts: &str) -> (String, Assoc) {
        (ts.to_string(), Assoc::Single)
    }
    fn use_paren(ta: (String, Assoc), assoc: Assoc) -> String {
        if ta.1 > assoc {
            format!("({})", ta.0)
        } else {
            ta.0
        }
    }
    fn gen_args<'a>(p: &'a syn::TypePath, nargs: usize, name: &str) -> Vec<&'a Type> {
        if let PathArguments::AngleBracketed(args) = &p.path.segments[0].arguments {
            if args.args.len() != nargs {
                panic!(
                    "{} expects {} generic(s) argument(s), found {}",
                    name,
                    nargs,
                    args.args.len()
                );
            }
            let mut result = Vec::new();
            for arg in &args.args {
                if let syn::GenericArgument::Type(tk) = arg {
                    result.push(tk);
                } else {
                    panic!("No type provided for {}", name);
                }
            }
            result
        } else {
            panic!("{} used with no generic arguments", name);
        }
    }

    fn ts_type_assoc(ty: &Type) -> (String, Assoc) {
        match ty {
            Type::Path(p) => match crate::join_path(&p.path).as_str() {
                "bool" => single("boolean"),
                "u64" => single("number"),
                "i8" | "u8" | "i16" | "u16" | "i32" | "u32" => single("number"),
                "String" => single("string"),
                "Option" => {
                    let targs = gen_args(p, 1, "Option");
                    let ta = ts_type_assoc(&targs[0]);
                    (format!("{}|null", use_paren(ta, Assoc::Or)), Assoc::Or)
                }
                "Vec" | "HashSet" | "BTreeSet" => {
                    let targs = gen_args(p, 1, "Vec");
                    let ta = ts_type_assoc(&targs[0]);
                    (format!("{}[]", use_paren(ta, Assoc::Vec)), Assoc::Vec)
                }
                "HashMap" | "BTreeMap" => {
                    let targs = gen_args(p, 2, "HashMap");
                    let (tks, _) = ts_type_assoc(&targs[0]);
                    let (tvs, _) = ts_type_assoc(&targs[1]);
                    (format!("Record<{}, {}>", tks, tvs), Assoc::Single)
                }
                s => single(s),
            },
            Type::Paren(paren) => ts_type_assoc(paren.elem.as_ref()),
            Type::Tuple(tuple) => {
                if tuple.elems.is_empty() {
                    ("void".into(), Assoc::Single)
                } else {
                    let mut tys = Vec::new();
                    for elem_type in &tuple.elems {
                        let (t, _) = ts_type_assoc(&elem_type);
                        tys.push(t);
                    }
                    (format!("[{}]", tys.join(", ")), Assoc::Single)
                }
            }
            Type::Reference(reference) => ts_type_assoc(&reference.elem),
            _ => panic!("type not supported: {:?}", ty),
        }
    }
    ts_type_assoc(ty).0
}

/// Returns the signature of the given Rust `method`.
/// The resulting TypeScript binding is a valid method definition expected by the NEAR RPC.
/// Thus, the following conversion are applied:
/// - Function arguments are packed into a single TypeScript object argument
/// - Return type is wrapped into a `Promise`
/// - Types are converted using `ts_type`
///
/// ### Examples
///
/// ```
/// use syn::parse_str;
/// use near_syn::ts::ts_sig;
///
/// assert_eq!(ts_sig(&parse_str("fn a() {}").unwrap()), "a(): Promise<void>;");
/// assert_eq!(ts_sig(&parse_str("fn b(x: U128) {}").unwrap()), "b(args: { x: U128 }): Promise<void>;");
/// assert_eq!(ts_sig(&parse_str("fn c(x: U128, y: String) -> Vec<Token> {}").unwrap()), "c(args: { x: U128, y: string }): Promise<Token[]>;");
/// assert_eq!(ts_sig(&parse_str("fn d(x: U128, y: String, z: Option<U64>) -> Vec<Token> {}").unwrap()), "d(args: { x: U128, y: string, z: U64|null }): Promise<Token[]>;");
/// assert_eq!(ts_sig(&parse_str("fn e(x: U128) -> () {}").unwrap()), "e(args: { x: U128 }): Promise<void>;");
/// assert_eq!(ts_sig(&parse_str("fn f(paren: (String)) {}").unwrap()), "f(args: { paren: string }): Promise<void>;");
/// assert_eq!(ts_sig(&parse_str("fn get(&self) -> u32 {}").unwrap()), "get(): Promise<number>;");
/// assert_eq!(ts_sig(&parse_str("fn set(&mut self) {}").unwrap()), "set(gas?: any): Promise<void>;");
/// assert_eq!(ts_sig(&parse_str("fn set_args(&mut self, x: u32) {}").unwrap()), "set_args(args: { x: number }, gas?: any): Promise<void>;");
/// assert_eq!(ts_sig(&parse_str("fn a() -> Promise {}").unwrap()), "a(): Promise<void>;");
/// ```
pub fn ts_sig(method: &ImplItemMethod) -> String {
    let mut args = Vec::new();
    for arg in method.sig.inputs.iter() {
        match arg {
            syn::FnArg::Typed(pat_type) => {
                if let syn::Pat::Ident(pat_ident) = pat_type.pat.deref() {
                    let type_name = ts_type(&pat_type.ty);
                    let arg_ident = &pat_ident.ident;
                    args.push(format!("{}: {}", arg_ident, type_name));
                }
            }
            _ => {}
        }
    }

    if method.is_init() {
        format!("{}: {{ {} }};", method.sig.ident, args.join(", "),)
    } else {
        let mut args_decl = Vec::new();
        if args.len() > 0 {
            args_decl.push(format!("args: {{ {} }}", args.join(", ")));
        };
        if method.is_mut() {
            args_decl.push("gas?: any".into());
        }
        if method.is_payable() {
            args_decl.push("amount?: any".into());
        }

        format!(
            "{}({}): Promise<{}>;",
            method.sig.ident,
            args_decl.join(", "),
            ts_ret_type(&method.sig.output),
        )
    }
}

/// Returns the TypeScript representation of output's type given the Rust `ret_type`.
/// The resulting TypeScript return type is a valid output type expected by the NEAR RPC.
/// Thus, the following conversion are applied:
/// - Types are converted using `ts_type`
/// - Return type of `Promise` is mapped to `void`.
///
/// ### Examples
///
/// ```
/// use syn::parse_str;
/// use near_syn::ts::ts_ret_type;
///
/// assert_eq!(ts_ret_type(&parse_str(" ").unwrap()), "void");
/// assert_eq!(ts_ret_type(&parse_str("-> Promise<u32>").unwrap()), "void");
/// assert_eq!(ts_ret_type(&parse_str("-> Vec<Token>").unwrap()), "Token[]");
/// assert_eq!(ts_ret_type(&parse_str("-> u32").unwrap()), "number");
pub fn ts_ret_type(ret_type: &ReturnType) -> String {
    match ret_type {
        ReturnType::Default => "void".into(),
        ReturnType::Type(_, typ) => {
            let ty = ts_type(typ.deref());
            match ty.as_str() {
                "Promise" | "PromiseOrValue" => "void".to_string(),
                _ => ty,
            }
        }
    }
}

#[cfg(test)]
mod tests {

    use crate::ts::ts_type;

    #[test]
    #[should_panic(expected = "Option used with no generic arg")]
    fn ts_type_on_option_with_no_args_should_panic() {
        ts_type(&syn::parse_str("Option").unwrap());
    }

    #[test]
    #[should_panic(expected = "Option expects 1 generic(s) argument(s), found 2")]
    fn ts_type_on_option_with_more_than_one_arg_should_panic() {
        ts_type(&syn::parse_str("Option<String, U128>").unwrap());
    }

    #[test]
    #[should_panic(expected = "Vec used with no generic arg")]
    fn ts_type_on_vec_with_no_args_should_panic() {
        ts_type(&syn::parse_str("Vec").unwrap());
    }

    #[test]
    #[should_panic(expected = "Vec expects 1 generic(s) argument(s), found 3")]
    fn ts_type_on_vec_with_more_than_one_arg_should_panic() {
        ts_type(&syn::parse_str("Vec<String, U128, u32>").unwrap());
    }

    #[test]
    #[should_panic(expected = "HashMap used with no generic arguments")]
    fn ts_type_on_hashmap_with_no_args_should_panic() {
        ts_type(&syn::parse_str("HashMap").unwrap());
    }

    #[test]
    #[should_panic(expected = "HashMap expects 2 generic(s) argument(s), found 1")]
    fn ts_type_on_hashmap_with_less_than_two_args_should_panic() {
        ts_type(&syn::parse_str("HashMap<U64>").unwrap());
    }
}
