#![deny(warnings)]

use chrono::Utc;
use clap::Parser;
use near_syn::{
    join_path,
    ts::{ts_sig, TS},
    write_docs, NearImpl, NearMethod,
};
use std::{
    env,
    fs::File,
    io::{self, Read},
    path::Path,
};
use syn::{ImplItem, Item::Impl, ItemImpl, Type};

/// Analyzes Rust source files to generate either TypeScript bindings or Markdown documentation
#[derive(Parser)]
#[clap(version = env!("CARGO_PKG_VERSION"), author = env!("CARGO_PKG_AUTHORS"))]
struct Args {
    #[clap(subcommand)]
    cmd: Cmd,
}

#[derive(Parser)]
enum Cmd {
    /// Emits TypeScript bindings
    #[clap(version = env!("CARGO_PKG_VERSION"), author = env!("CARGO_PKG_AUTHORS"))]
    TS(EmitArgs),

    /// Emits Markdown documentation
    #[clap(version = env!("CARGO_PKG_VERSION"), author = env!("CARGO_PKG_AUTHORS"))]
    MD(EmitArgs),
}

#[derive(Parser)]
struct EmitArgs {
    /// Does not emit date/time information,
    /// otherwise emits current time
    #[clap(long)]
    no_now: bool,

    /// Rust source files (*.rs) to analize
    #[clap()]
    files: Vec<String>,
}

fn main() {
    let args = Args::parse();

    match args.cmd {
        Cmd::TS(args) => emit_ts(args),
        Cmd::MD(args) => emit_md(args),
    }
}

fn emit_ts(args: EmitArgs) {
    let mut ts = TS::new(std::io::stdout());
    ts.ts_prelude(
        if args.no_now {
            "".to_string()
        } else {
            format!(" on {}", Utc::now())
        },
        env!("CARGO_BIN_NAME"),
    );

    for file_name in args.files {
        let ast = parse_rust(file_name);
        ts.ts_items(&ast.items);
    }
    ts.ts_extend_traits();
    ts.ts_contract_methods();
}

fn emit_md(args: EmitArgs) {
    let utc_now = Utc::now();
    println!(
        "<!-- AUTOGENERATED doc{}, do not modify! -->",
        if args.no_now {
            "".to_string()
        } else {
            format!(" on {}", utc_now)
        }
    );
    println!("# Contract\n");

    for file_name in &args.files {
        let ast = parse_rust(file_name);
        md(&ast);
    }

    println!("\n---\n\nReferences\n");
    println!("- :rocket: Initialization method. Needs to be called right after deployment.");
    println!("- :eyeglasses: View only method, *i.e.*, does not modify the contract state.");
    println!("- :writing_hand: Call method, i.e., does modify the contract state.");
    println!("- &#x24C3; Payable method, i.e., call needs to have an attached NEAR deposit.");

    println!(
        "\n---\n\n*This documentation was generated with* **{} v{}** <{}>{}",
        env!("CARGO_BIN_NAME"),
        env!("CARGO_PKG_VERSION"),
        env!("CARGO_PKG_REPOSITORY"),
        if args.no_now {
            "".to_string()
        } else {
            format!(" *on {}*", utc_now)
        }
    );
}

fn md(syntax: &syn::File) {
    write_docs(&mut io::stdout(), &syntax.attrs, |l| l.trim().to_string());

    for item in &syntax.items {
        if let Impl(impl_item) = item {
            if impl_item.is_bindgen() && impl_item.has_exported_methods() {
                if let Some((_, trait_path, _)) = &impl_item.trait_ {
                    println!("\n## Methods for `{}` interface", join_path(trait_path));
                } else {
                    if let Type::Path(type_path) = &*impl_item.self_ty {
                        println!("\n## Methods for {}", join_path(&type_path.path));
                    } else {
                        println!("\n## Methods for Contract");
                    }
                }

                methods(&impl_item);
            }
        }
    }
}

fn methods(input: &ItemImpl) {
    for impl_item in input.items.iter() {
        if let ImplItem::Method(method) = impl_item {
            if method.is_exported(input) {
                let mut mut_mod = if method.is_mut() {
                    if method.is_payable() {
                        "&#x24C3;"
                    } else {
                        ":writing_hand:"
                    }
                } else {
                    ":eyeglasses:"
                };
                let init_decl = if method.is_init() {
                    mut_mod = ":rocket:";
                    " (*constructor*)"
                } else {
                    ""
                };
                println!("\n### {} `{}`{}\n", mut_mod, method.sig.ident, init_decl);
                println!("```typescript\n{}\n```\n", ts_sig(&method));
                write_docs(&mut io::stdout(), &method.attrs, |l| l.trim().to_string());
            }
        }
    }
}

/// Returns the Rust syntax tree for the given `file_name` path.
/// Panics if the file cannot be open or the file has syntax errors.
///
/// ## Example
///
/// ```no_run
/// let mut ts = near_syn::ts::TS::new(std::io::stdout());
/// let ast = near_syn::parse_rust("path/to/file.rs");
/// ts.ts_items(&ast.items);
/// ```
fn parse_rust<S: AsRef<Path>>(file_name: S) -> syn::File {
    let mut file = File::open(file_name).expect("Unable to open file");
    let mut src = String::new();
    file.read_to_string(&mut src).expect("Unable to read file");

    syn::parse_file(&src).expect("Unable to parse file")
}
