use std::collections::HashMap;
use std::sync::Arc;

use rand::seq::SliceRandom;
use rand::Rng;

use crate::db::TestDB;
use crate::{ShardTries, Store};
use near_primitives_v01::account::id::AccountId;
use near_primitives_v01::hash::CryptoHash;
use near_primitives_v01::receipt::{DataReceipt, Receipt, ReceiptEnum};
use near_primitives_v01::shard_layout::{ShardUId, ShardVersion};
use near_primitives_v01::types::NumShards;
use std::str::from_utf8;

/// Creates an in-memory database.
pub fn create_test_store() -> Arc<Store> {
    let db = Arc::pin(TestDB::new());
    Arc::new(Store::new(db))
}

/// Creates a Trie using an in-memory database.
pub fn create_tries() -> ShardTries {
    let store = create_test_store();
    ShardTries::new(store, 0, 1)
}

pub fn create_tries_complex(shard_version: ShardVersion, num_shards: NumShards) -> ShardTries {
    let store = create_test_store();
    ShardTries::new(store, shard_version, num_shards)
}

pub fn test_populate_trie(
    tries: &ShardTries,
    root: &CryptoHash,
    shard_uid: ShardUId,
    changes: Vec<(Vec<u8>, Option<Vec<u8>>)>,
) -> CryptoHash {
    let trie = tries.get_trie_for_shard(shard_uid);
    assert_eq!(trie.storage.as_caching_storage().unwrap().shard_uid.shard_id, 0);
    let trie_changes = trie.update(root, changes.iter().cloned()).unwrap();
    let (store_update, root) = tries.apply_all(&trie_changes, shard_uid).unwrap();
    store_update.commit().unwrap();
    let deduped = simplify_changes(&changes);
    for (key, value) in deduped {
        assert_eq!(trie.get(&root, &key), Ok(value));
    }
    root
}

pub fn gen_accounts(rng: &mut impl Rng, max_size: usize) -> Vec<AccountId> {
    let alphabet = b"abcdefghijklmn";
    let size = rng.gen_range(0, max_size) + 1;

    let mut accounts = vec![];
    for _ in 0..size {
        let str_length = rng.gen_range(4, 8);
        let s: Vec<u8> = (0..str_length).map(|_| alphabet.choose(rng).unwrap().clone()).collect();
        let account_id: AccountId = from_utf8(&s).unwrap().parse().unwrap();
        accounts.push(account_id);
    }
    accounts
}

pub fn gen_receipts(rng: &mut impl Rng, max_size: usize) -> Vec<Receipt> {
    let accounts = gen_accounts(rng, max_size);
    accounts
        .iter()
        .map(|account_id| Receipt {
            predecessor_id: account_id.clone(),
            receiver_id: account_id.clone(),
            receipt_id: CryptoHash::default(),
            receipt: ReceiptEnum::Data(DataReceipt { data_id: CryptoHash::default(), data: None }),
        })
        .collect()
}

pub fn gen_changes(rng: &mut impl Rng, max_size: usize) -> Vec<(Vec<u8>, Option<Vec<u8>>)> {
    let alphabet = &b"abcdefgh"[0..rng.gen_range(2, 8)];
    let max_length = rng.gen_range(2, 8);

    let mut state: HashMap<Vec<u8>, Vec<u8>> = HashMap::new();
    let mut result = Vec::new();
    let delete_probability = rng.gen_range(0.1, 0.5);
    let size = rng.gen_range(0, max_size) + 1;
    for _ in 0..size {
        let key_length = rng.gen_range(1, max_length);
        let key: Vec<u8> = (0..key_length).map(|_| alphabet.choose(rng).unwrap().clone()).collect();

        let delete = rng.gen_range(0.0, 1.0) < delete_probability;
        if delete {
            let mut keys: Vec<_> = state.keys().cloned().collect();
            keys.push(key);
            let key = keys.choose(rng).unwrap().clone();
            state.remove(&key);
            result.push((key.clone(), None));
        } else {
            let value_length = rng.gen_range(1, max_length);
            let value: Vec<u8> =
                (0..value_length).map(|_| alphabet.choose(rng).unwrap().clone()).collect();
            result.push((key.clone(), Some(value.clone())));
            state.insert(key, value);
        }
    }
    result
}

pub(crate) fn simplify_changes(
    changes: &Vec<(Vec<u8>, Option<Vec<u8>>)>,
) -> Vec<(Vec<u8>, Option<Vec<u8>>)> {
    let mut state: HashMap<Vec<u8>, Vec<u8>> = HashMap::new();
    for (key, value) in changes.iter() {
        if let Some(value) = value {
            state.insert(key.clone(), value.clone());
        } else {
            state.remove(key);
        }
    }
    let mut result: Vec<_> = state.into_iter().map(|(k, v)| (k, Some(v))).collect();
    result.sort();
    result
}
