//! Instances for documentation tests or tests.

use crate::{coo::CooArray, csl::CslArray};
#[cfg(feature = "alloc")]
use crate::{coo::CooVec, csl::CslVec};

/// As odd as it may seem, this illustration is just a guide to get a grasp of
/// a 5D structure.
///
/// ```rust
/// //          ___ ___ ___            ___ ___ ___            ___ ___ ___
/// //        /   /   /   /\         / 3 /   /   /\         /   /   /   /\
/// //       /___/___/___/ /\       /_3_/___/___/ /\       /___/___/___/ /\
/// //      /   /   /   /\/ /\     /   /   /   /\/ /\     /   / 4 /   /\/ /\
/// //     /___/___/___/ /\/ /    /___/___/___/ /\/ /    /___/_4_/___/ /\/ /
/// //    /   /   /   /\/ /\/    /   /   /   /\/ /\/    /   /   /   /\/ /\/
/// //   /___/___/___/ /\/ /    /___/___/___/ /\/ /    /___/___/___/ /\/ /
/// //  /   /   /   /\/1/\/    /   /   /   /\/ /\/    /   /   /   /\/ /\/
/// // /___/___/___/ /\/ /    /___/___/___/ /\/ /    /___/___/___/ /\/ /
/// // \___\___\___\/ /\/     \___\___\___\/ /\/     \___\___\___\/ /\/
/// //  \___\___\___\/ /       \___\_2_\___\/ /       \___\___\___\/ /
/// //   \___\___\___\/         \___\___\___\/         \___\___\___\/
/// //
/// //          ___ ___ ___            ___ ___ ___            ___ ___ ___
/// //        /   /   /   /\         /   /   /   /\         /   /   / 6 /\
/// //       /___/___/___/ /\       /___/___/___/ /\       /___/___/_6_/6/\
/// //      /   /   /   /\/ /\     /   /   /   /\/ /\     /   /   /   /\/ /\
/// //     /___/___/___/ /\/ /    /___/___/___/ /\/ /    /___/___/___/ /\/7/
/// //    /   /   /   /\/ /\/    /   /   /   /\/ /\/    /   /   /   /\/ /\/
/// //   /___/___/___/ /\/ /    /___/___/___/ /\/ /    /___/___/___/ /\/ /
/// //  /   /   /   /\/ /\/    /   /   /   /\/ /\/    /   /   /   /\/ /\/
/// // /___/___/___/ /\/ /    /___/___/___/ /\/ /    /___/___/___/ /\/ /
/// // \___\___\___\/ /\/     \___\___\___\/ /\/     \___\___\___\/ /\/
/// //  \___\___\___\/ /       \___\___\___\/ /       \___\___\___\/ /
/// //   \___\___\___\/         \___\_5_\___\/         \___\___\___\/
/// use ndsparse::coo::CooArray;
/// let _ = CooArray::new(
///   [2, 3, 4, 3, 3],
///   [
///     ([0, 0, 1, 1, 2], 1),
///     ([0, 1, 0, 1, 1], 2),
///     ([0, 1, 3, 0, 0], 3),
///     ([0, 2, 2, 0, 1], 4),
///     ([1, 1, 0, 2, 1], 5),
///     ([1, 2, 3, 0, 2], 6),
///     ([1, 2, 3, 2, 2], 7),
///   ],
/// );
/// ```
#[inline]
pub fn coo_array_5() -> CooArray<i32, 5, 7> {
  CooArray {
    dims: [2, 3, 4, 3, 3],
    data: [
      ([0, 0, 1, 1, 2], 1),
      ([0, 1, 0, 1, 1], 2),
      ([0, 1, 3, 0, 0], 3),
      ([0, 2, 2, 0, 1], 4),
      ([1, 1, 0, 2, 1], 5),
      ([1, 2, 3, 0, 2], 6),
      ([1, 2, 3, 2, 2], 7),
    ],
  }
}

/// [`Vec`](alloc::vec::Vec) version of [`coo_array_5`].
#[cfg(feature = "alloc")]
#[inline]
pub fn coo_vec_5() -> CooVec<i32, 5> {
  let coo = coo_array_5();
  CooVec { dims: coo.dims, data: coo.data.to_vec() }
}

/// Two cuboids illustrating a [2, 3, 4, 5] 4D in a [w, y, z, x] order, i.e., each "line"
/// or 1D representation is a left to right row and each "matrix" or 2D representation
/// is filled in a top-down manner.
///
/// ```rust
/// // w: left to right
/// // y: top to bottom
/// // z: front to back
/// // x: left to right
/// //
/// //          ___ ___ ___ ___ ___            ___ ___ ___ ___ ___
/// //        /   /   /   / 4 / 5 /\         /   /   /   /   /   /\
/// //       /___/___/___/_4_/_5_/5/\       /___/___/___/___/___/ /\
/// //      /   /   /   /   /   /\/ /\     /   /   / 9 /   /   /\/ /\
/// //     /___/___/___/___/___/ /\/ /    /___/___/_9_/___/___/ /\/ /
/// //    /   / 3 /   /   /   /\/ /\/    /   /   /   /   /   /\/ /\/
/// //   /___/_3_/___/___/___/ /\/ /    /___/_ _/___/___/___/ /\/ /
/// //  / 1 /   /   / 2 /   /\/ /\/    /   /   /   /   /   /\/ /\/
/// // /_1_/___/___/_2_/___/ /\/8/    /___/___/___/___/___/ /\/ /
/// // \_1_\___\___\_2_\___\/ /\/     \___\___\___\___\___\/ /\/
/// //  \___\___\_6_\___\___\/ /       \___\___\___\___\___\/ /
/// //   \___\___\_7_\___\___\/         \___\___\___\___\___\/
/// use ndsparse::csl::CslArray;
/// let _ = CslArray::new(
///   [2, 3, 4, 5],
///   [1, 2, 3, 4, 5, 6, 7, 8, 9],
///   [0, 3, 1, 3, 4, 2, 2, 4, 2],
///   [0, 2, 3, 3, 5, 6, 6, 6, 6, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9],
/// );
/// ```
#[inline]
pub fn csl_array_4() -> CslArray<i32, 4, 9, 25> {
  CslArray {
    data: [1, 2, 3, 4, 5, 6, 7, 8, 9],
    dims: [2, 3, 4, 5],
    indcs: [0, 3, 1, 3, 4, 2, 2, 4, 2],
    offs: [0, 2, 3, 3, 5, 6, 6, 6, 6, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9],
  }
}

/// [`Vec`](alloc::vec::Vec) version of [`csl_array_4`].
#[cfg(feature = "alloc")]
#[inline]
pub fn csl_vec_4() -> CslVec<i32, 4> {
  let csl = csl_array_4();
  CslVec {
    data: csl.data.to_vec(),
    dims: csl.dims,
    indcs: csl.indcs.to_vec(),
    offs: csl.offs.to_vec(),
  }
}
