use std::sync::Arc;
use std::{cmp::Ordering, iter::Iterator};

use log::debug;
use rspotify::model::playlist::{FullPlaylist, SimplifiedPlaylist};

use crate::playable::Playable;
use crate::queue::Queue;
use crate::spotify::Spotify;
use crate::track::Track;
use crate::traits::{IntoBoxedViewExt, ListItem, ViewExt};
use crate::ui::playlist::PlaylistView;
use crate::{command::SortDirection, command::SortKey, library::Library};

#[derive(Clone, Debug, Deserialize, Serialize)]
pub struct Playlist {
    pub id: String,
    pub name: String,
    pub owner_id: String,
    pub snapshot_id: String,
    pub num_tracks: usize,
    pub tracks: Option<Vec<Track>>,
    pub collaborative: bool,
}

impl Playlist {
    pub fn load_tracks(&mut self, spotify: Spotify) {
        if self.tracks.is_some() {
            return;
        }

        self.tracks = Some(self.get_all_tracks(spotify));
    }

    fn get_all_tracks(&self, spotify: Spotify) -> Vec<Track> {
        let tracks_result = spotify.user_playlist_tracks(&self.id);
        while !tracks_result.at_end() {
            tracks_result.next();
        }

        let tracks = tracks_result.items.read().unwrap();
        tracks.clone()
    }

    pub fn has_track(&self, track_id: &str) -> bool {
        self.tracks.as_ref().map_or(false, |tracks| {
            tracks
                .iter()
                .any(|track| track.id == Some(track_id.to_string()))
        })
    }

    pub fn delete_track(&mut self, index: usize, spotify: Spotify, library: Arc<Library>) -> bool {
        let track = self.tracks.as_ref().unwrap()[index].clone();
        debug!("deleting track: {} {:?}", index, track);
        match spotify.delete_tracks(&self.id, &self.snapshot_id, &[(&track, track.list_index)]) {
            false => false,
            true => {
                if let Some(tracks) = &mut self.tracks {
                    tracks.remove(index);
                    library.playlist_update(&self);
                }

                true
            }
        }
    }

    pub fn append_tracks(&mut self, new_tracks: &[Track], spotify: Spotify, library: Arc<Library>) {
        let track_ids: Vec<String> = new_tracks
            .to_vec()
            .iter()
            .filter_map(|t| t.id.clone())
            .collect();

        let mut has_modified = false;

        if spotify.append_tracks(&self.id, &track_ids, None) {
            if let Some(tracks) = &mut self.tracks {
                tracks.append(&mut new_tracks.to_vec());
                has_modified = true;
            }
        }

        if has_modified {
            library.playlist_update(self);
        }
    }

    pub fn sort(&mut self, key: &SortKey, direction: &SortDirection) {
        fn compare_artists(a: Vec<String>, b: Vec<String>) -> Ordering {
            let sanitize_artists_name = |x: Vec<String>| -> Vec<String> {
                x.iter()
                    .map(|x| {
                        x.to_lowercase()
                            .split(' ')
                            .skip_while(|x| x == &"the")
                            .collect()
                    })
                    .collect()
            };

            let a = sanitize_artists_name(a);
            let b = sanitize_artists_name(b);

            a.cmp(&b)
        }

        if let Some(c) = self.tracks.as_mut() {
            c.sort_by(|a, b| match (a.track(), b.track()) {
                (Some(a), Some(b)) => match (key, direction) {
                    (SortKey::Title, SortDirection::Ascending) => {
                        a.title.to_lowercase().cmp(&b.title.to_lowercase())
                    }
                    (SortKey::Title, SortDirection::Descending) => {
                        b.title.to_lowercase().cmp(&a.title.to_lowercase())
                    }
                    (SortKey::Duration, SortDirection::Ascending) => a.duration.cmp(&b.duration),
                    (SortKey::Duration, SortDirection::Descending) => b.duration.cmp(&a.duration),
                    (SortKey::Album, SortDirection::Ascending) => a
                        .album
                        .map(|x| x.to_lowercase())
                        .cmp(&b.album.map(|x| x.to_lowercase())),
                    (SortKey::Album, SortDirection::Descending) => b
                        .album
                        .map(|x| x.to_lowercase())
                        .cmp(&a.album.map(|x| x.to_lowercase())),
                    (SortKey::Added, SortDirection::Ascending) => a.added_at.cmp(&b.added_at),
                    (SortKey::Added, SortDirection::Descending) => b.added_at.cmp(&a.added_at),
                    (SortKey::Artist, SortDirection::Ascending) => {
                        compare_artists(a.artists, b.artists)
                    }
                    (SortKey::Artist, SortDirection::Descending) => {
                        compare_artists(b.artists, a.artists)
                    }
                },
                _ => std::cmp::Ordering::Equal,
            })
        }
    }
}

impl From<&SimplifiedPlaylist> for Playlist {
    fn from(list: &SimplifiedPlaylist) -> Self {
        let num_tracks = if let Some(number) = list.tracks.get("total") {
            number.as_u64().unwrap() as usize
        } else {
            0
        };

        Playlist {
            id: list.id.clone(),
            name: list.name.clone(),
            owner_id: list.owner.id.clone(),
            snapshot_id: list.snapshot_id.clone(),
            num_tracks,
            tracks: None,
            collaborative: list.collaborative,
        }
    }
}

impl From<&FullPlaylist> for Playlist {
    fn from(list: &FullPlaylist) -> Self {
        Playlist {
            id: list.id.clone(),
            name: list.name.clone(),
            owner_id: list.owner.id.clone(),
            snapshot_id: list.snapshot_id.clone(),
            num_tracks: list.tracks.total as usize,
            tracks: None,
            collaborative: list.collaborative,
        }
    }
}

impl ListItem for Playlist {
    fn is_playing(&self, queue: Arc<Queue>) -> bool {
        if let Some(tracks) = self.tracks.as_ref() {
            let playing: Vec<String> = queue
                .queue
                .read()
                .unwrap()
                .iter()
                .filter_map(|t| t.id())
                .collect();
            let ids: Vec<String> = tracks.iter().filter_map(|t| t.id.clone()).collect();
            !ids.is_empty() && playing == ids
        } else {
            false
        }
    }

    fn as_listitem(&self) -> Box<dyn ListItem> {
        Box::new(self.clone())
    }

    fn display_left(&self) -> String {
        self.name.clone()
    }

    fn display_right(&self, library: Arc<Library>) -> String {
        let saved = if library.is_saved_playlist(self) {
            if library.cfg.values().use_nerdfont.unwrap_or(false) {
                "\u{f62b} "
            } else {
                "✓ "
            }
        } else {
            ""
        };

        let num_tracks = self
            .tracks
            .as_ref()
            .map(|t| t.len())
            .unwrap_or(self.num_tracks);

        format!("{}{:>4} tracks", saved, num_tracks)
    }

    fn play(&mut self, queue: Arc<Queue>) {
        self.load_tracks(queue.get_spotify());

        if let Some(tracks) = &self.tracks {
            let tracks: Vec<Playable> = tracks
                .iter()
                .map(|track| Playable::Track(track.clone()))
                .collect();
            let index = queue.append_next(tracks);
            queue.play(index, true, true);
        }
    }

    fn play_next(&mut self, queue: Arc<Queue>) {
        self.load_tracks(queue.get_spotify());

        if let Some(tracks) = self.tracks.as_ref() {
            for track in tracks.iter().rev() {
                queue.insert_after_current(Playable::Track(track.clone()));
            }
        }
    }

    fn queue(&mut self, queue: Arc<Queue>) {
        self.load_tracks(queue.get_spotify());

        if let Some(tracks) = self.tracks.as_ref() {
            for track in tracks.iter() {
                queue.append(Playable::Track(track.clone()));
            }
        }
    }

    fn save(&mut self, library: Arc<Library>) {
        library.follow_playlist(self);
    }

    fn unsave(&mut self, library: Arc<Library>) {
        library.delete_playlist(&self.id);
    }

    fn toggle_saved(&mut self, library: Arc<Library>) {
        // Don't allow users to unsave their own playlists with one keypress
        if !library.is_followed_playlist(self) {
            return;
        }

        if library.is_saved_playlist(self) {
            library.delete_playlist(&self.id);
        } else {
            library.follow_playlist(self);
        }
    }

    fn open(&self, queue: Arc<Queue>, library: Arc<Library>) -> Option<Box<dyn ViewExt>> {
        Some(PlaylistView::new(queue, library, self).into_boxed_view_ext())
    }

    fn share_url(&self) -> Option<String> {
        Some(format!(
            "https://open.spotify.com/user/{}/playlist/{}",
            self.owner_id, self.id
        ))
    }
}
