use config::{Config, File, FileFormat};
use serde::{Serialize, Serializer};
use std::fs::File as FsFile;
use std::io::Write;
use std::path::PathBuf;
use url::Url;

fn url_serialize<S>(url: &Url, s: S) -> Result<S::Ok, S::Error>
where
    S: Serializer,
{
    s.serialize_str(&url.to_string())
}

#[derive(Clone, Debug, serde::Serialize)]
pub struct Bookmark {
    pub title: String,
    #[serde(serialize_with = "url_serialize")]
    pub url: Url,
    pub tags: Vec<String>,
}

#[derive(Clone, Debug, Serialize)]
pub struct Bookmarks {
    /// All bookmarks
    pub entries: Vec<Bookmark>,
}

impl Bookmarks {
    pub fn new() -> Bookmarks {
        let mut s = Config::new();
        let confdir = Bookmarks::get_bookmark_path();
        println!("Looking for bookmarks file {:?}", confdir);
        if confdir.as_path().exists() {
            match s.merge(File::new(
                confdir.to_str().expect("non-UTF8 bookmarks filename"),
                FileFormat::Toml,
            )) {
                Ok(_s) => (),
                Err(e) => {
                    println!("Could not read bookmarks file: {}", e);
                }
            }
        }
        let mut entries = Vec::new();
        println!("Reading bookmarks...");
        info!("bookmarks: {:?}", s.get_array("bookmark"));
        if let Ok(e) = s.get_array("bookmark") {
            for value in e {
                if let Ok(v) = value.into_table() {
                    let url = if let Ok(url) = v["url"].clone().into_str() {
                        url
                    } else {
                        continue;
                    };
                    let title = if let Ok(title) = v["title"].clone().into_str() {
                        title
                    } else {
                        continue;
                    };
                    if let Ok(u) = Url::parse(&url) {
                        let h = Bookmark {
                            url: u.clone(),
                            title,
                            tags: Vec::new(),
                        };
                        entries.push(h);
                    }
                }
            }
        }
        Bookmarks { entries }
    }

    fn get_bookmark_path() -> PathBuf {
        let mut dir = dirs::config_dir().expect("no configuration directory");
        dir.push(env!("CARGO_PKG_NAME"));
        dir.push("bookmarks");
        info!("Looking for bookmark file {:?}", dir);
        dir
    }

    /// Replace an existting bookmark or add a new bookmark.
    /// If an entry is replaced, it will remain at the same position
    /// Returns the index of the existing entry or None.
    pub fn insert(&mut self, entry: Bookmark) -> Option<usize> {
        info!("Adding entry to bookmark: {:?}", entry);
        let index = self.entries.iter().position(|e| e.url == entry.url);
        if let Some(i) = index {
            // replace item
            self.entries.remove(i);
            self.entries.insert(i, entry);
        } else {
            // insert new item at end
            self.entries.push(entry);
        };
        self.write_bookmarks_to_file()
            .unwrap_or_else(|err| warn!("Could not write bookmarks file: {}", err));
        index
    }

    pub fn remove(&mut self, url: &Url) {
        info!("Removing entry to bookmark: {:?}", url);
        self.entries.retain(|e| &e.url != url);
        match self.write_bookmarks_to_file() {
            Err(why) => warn!("Could not write bookmarks file: {}", why),
            Ok(()) => (),
        }
    }

    pub fn get_bookmarks(&self) -> Vec<Bookmark> {
        self.entries.clone()
    }

    pub fn write_bookmarks_to_file(&mut self) -> std::io::Result<()> {
        let path = Bookmarks::get_bookmark_path();
        info!("Saving bookmarks to file: {:?}", path);

        let mut file = match FsFile::create(&path) {
            Err(why) => return Err(why),
            Ok(file) => file,
        };

        file.write_all(b"# Automatically generated by ncgopher.\n")?;
        for b in self.clone().entries {
            file.write_all(b"\n[[bookmark]]\n")?;
            let item = toml::to_string(&b).unwrap();
            file.write_all(item.as_bytes())?;
        }
        Ok(())
    }
}
