use std::default::Default;
use std::io::Read;
use std::fs::File;
use std::fmt::{Display,Formatter};
use sha2::{Sha256, Digest};

#[derive(Debug)]
pub struct Builder {
    metadata: Vec<(String, String)>,
    files: Vec<(String, File)>,
}

impl Default for Builder {
    fn default() -> Self {
        Builder::new()
    }
}

impl Builder {
    pub fn new() -> Self {
        let metadata = Vec::new();
        let files = Vec::new();
        Self { metadata, files }
    }

    pub fn add_file<T: Into<String>>(&mut self, name: T, file: File) -> &mut Self {
        let pair = (name.into(), file);
        self.files.push(pair);
        self
    }

    pub fn add_metadata<T: Into<String>, U: Into<String>>(&mut self, key: T, value: U) -> &mut Self {
        let pair = (key.into(), value.into() );
        self.metadata.push(pair);
        self
    }

    pub fn build(self) -> Metadata {
        Metadata {
            metadata: self.metadata,
            files: self.files,
        }

    }
}

pub struct Metadata {
    metadata: Vec<(String, String)>,
    files: Vec<(String, File)>,
}

impl Metadata {
    fn render_metadata(&self) -> String {
        self.metadata
        .iter()
        .map(|ref item| Self::render_pair(&item.0, &item.1))
        .collect::<Vec<String>>()
        .join("\n")
    }

    fn render_files(&self) -> String {
        self.files
        .iter()
        .map(|ref mut item| Self::render_file_pair(&item.0, &item.1))
        .collect::<Vec<String>>()
        .join("\n")
    }

    fn render_pair<T: Display, U: Display>(key: T, value: U) -> String {
        format!("{}: {}", key, value)
    }

    fn render_file_pair(filename: &str, mut file: &File) -> String {
        let mut buffer = Vec::new();
        file.read_to_end(&mut buffer).expect("Failed to read file");
        let shasum = format!("{:#x}", Sha256::digest(&buffer));
        Self::render_pair(filename, shasum)
    }
}

impl Display for Metadata {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), std::fmt::Error> {
        if self.metadata.is_empty() && self.files.is_empty() {
            write!(f, "")
        } else if !self.metadata.is_empty() && self.files.is_empty() {
            writeln!(f, "{}", self.render_metadata())
        } else if self.metadata.is_empty() && !self.files.is_empty() {
            writeln!(f, "{}", self.render_files())
        } else {
            writeln!(f, "{}\n\n{}", self.render_metadata(), self.render_files())
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use std::env;

    #[test]
    fn new() {
        let builder = Builder::new();
        assert!(&builder.metadata.is_empty());
        assert!(&builder.files.is_empty());
    }

    #[test]
    fn add_new_file() {
        let path = env::temp_dir().join("add_new_file");
        let file = File::create(path).unwrap();
        let mut builder = Builder::new();
        builder.add_file("foo", file);
        assert_eq!(builder.files.len(), 1);
        assert_eq!(builder.files.first().unwrap().0, "foo");
        assert!(builder.metadata.is_empty());
    }

    #[test]
    fn add_metadata() {
        let mut builder = Builder::new();
        builder.add_metadata("foo", "bar");
        assert_eq!(builder.metadata.len(), 1);
        assert_eq!(builder.metadata.first().unwrap().0, "foo");
        assert_eq!(builder.metadata.first().unwrap().1, "bar");
        assert!(builder.files.is_empty());
    }

    #[test]
    fn build() {
        let path = env::temp_dir().join("add_new_file");
        let file = File::create(path).unwrap();
        let mut builder = Builder::new();
        builder.add_file("file.txt", file);
        builder.add_metadata("foo", "bar");
        let metadata = builder.build();
        assert_eq!(metadata.metadata.len(), 1);
        assert_eq!(metadata.files.len(), 1);
    }

    #[test]
    fn display_with_no_files() {
        let metadata = Builder::new().build();
        assert_eq!(format!("{}", metadata), "");
    }

    #[test]
    fn display_with_no_metadata_or_files() {
        let mut builder = Builder::new();
        builder.add_metadata("foo", "bar");
        let metadata = builder.build();
        assert_eq!(format!("{}", metadata), "foo: bar\n");
    }

    #[test]
    fn display_with_no_metadata() {
        let path = env::temp_dir().join("add_new_file");
        let file = File::open(path).unwrap();
        let mut builder = Builder::new();
        builder.add_file("file.txt", file);
        let metadata = builder.build();
        assert_eq!(
            format!("{}", metadata),
            "file.txt: e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855\n"
        );
    }

    #[test]
    fn display_with_metadata_and_files() {
        let path = env::temp_dir().join("add_new_file");
        let file = File::open(path).unwrap();
        let mut builder = Builder::new();
        builder.add_metadata("foo", "bar");
        builder.add_file("file.txt", file);
        let metadata = builder.build();
        assert_eq!(
            format!("{}", metadata),
            "foo: bar\n\nfile.txt: e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855\n"
        );
    }
}

