use clap::Arg;
use std::path::{Path,PathBuf};
use std::io::Write;
use clap::App;
use url::Url;
use naver::Volume;
use naver_dl_metadata;
use thiserror::Error;
use downloader::{Download, Downloader};
use std::fs::{self,File};

#[derive(Debug, Error)]
enum Error {
    #[error("failure")]
    GeneralError,
    #[error("invalid url")]
    UrlError(#[from] url::ParseError),
    #[error("download error")]
    NaverError(#[from] naver::Error),
    #[error("could not find volumeNo query parameter in url")]
    MissingVolumeNoParamError,
    #[error("download fialed")]
    DownloadError(#[from] downloader::Error),
    #[error("io error")]
    IOError(#[from] std::io::Error),
    #[error("image error")]
    ImageError(#[from] image::ImageError),
}

fn main() -> Result<(), Error> {
    let matches = App::new(env!("CARGO_PKG_NAME"))
        .about(env!("CARGO_PKG_DESCRIPTION"))
        .author(env!("CARGO_PKG_AUTHORS"))
        .version(env!("CARGO_PKG_VERSION"))
        .arg(Arg::with_name("input").required(true).takes_value(true).help("Input url or volume id"))
        .arg(Arg::with_name("output").required(false).takes_value(true).help("Output directory"))
        .arg(Arg::with_name("skip-library").long("skip-library").takes_value(false).help("Skip library based directory structure"))
        .arg(Arg::with_name("skip-metadata").long("skip-metadata").takes_value(false).help("Skip creation of metadata file"))
        .arg(Arg::with_name("force").short("f").long("force").takes_value(false).help("Force downloading of existing images"))
        .get_matches();

    let url = Url::parse(matches.value_of("input").or(Some("")).unwrap())
        .or(Url::parse(&format!("https://post.naver.com/viewer/postView.nhn?volumeNo={}", matches.value_of("input").unwrap())))?;

    let id = url.query_pairs().find(|item| item.0.eq("volumeNo"))
        .ok_or_else(|| Error::MissingVolumeNoParamError)?.1;

    let volume = Volume::get(id)?;

    let mut dest = match matches.value_of("output") {
        Some(path) => {
            let path = Path::new(path);
            if !path.is_dir() { fs::create_dir_all(path)? }
            PathBuf::from(path)
        },
        None => std::env::current_dir()?
    };

    if !matches.is_present("skip-library") {
        dest = dest.join(&volume.member).join(&volume.series).join(&volume.id);
        fs::create_dir_all(&dest)?;
    }
    
    let mut downloader = Downloader::builder()
        .download_folder(&dest)
        .parallel_requests(5)
        .build()
        .map_err(|_| Error::GeneralError)?;

    let mut downloads: Vec<Download> = Vec::with_capacity(volume.images.len());

    let mut images: Vec<(String,File)> = Vec::with_capacity(volume.images.len());

    for (index, image) in volume.images.iter().enumerate() {
        let url = Url::parse(&image.url).unwrap();
        let extension = url.path().split(".").last().unwrap();
        let filename = format!("{}{}{}{:0>4}.{}", &volume.member, &volume.series, &volume.id, index + 1, extension);
        let path = &dest.join(&filename);

        let existing = image::image_dimensions(&path);

        if existing.is_ok() && !matches.is_present("force") {
            images.push((filename, File::open(path)?));
        } else {
            if Path::exists(&path) { fs::remove_file(&path)? };
            downloads.push(Download::new(&image.url).file_name(&path));
        }

    }

    let downloads = downloader.download(&downloads)?;

    for image in downloads {
        let filename = image.unwrap().file_name;
        images.push((filename.as_path().file_name().unwrap().to_str().unwrap().to_string(), File::open(&filename)?));
    }

    if !matches.is_present("skip-metadata") {
        let path = &dest.join("metadata.txt");
        if Path::exists(&path) && matches.is_present("force") { fs::remove_file(&path)? };

        println!("Writing metadata...");

        let mut metadata = naver_dl_metadata::Builder::new();

        for (filename, image) in images { metadata.add_file(filename, image); }

        metadata.add_metadata("agent", format!("naver-dl-{}", env!("CARGO_PKG_VERSION")));
        metadata.add_metadata("id", format!("{}/{}/{}", &volume.member, &volume.series, &volume.id));
        metadata.add_metadata("url", format!("https://post.naver.com/viewer/postView.nhn?volumeNo={}", &volume.id));
        metadata.add_metadata("timestamp", format!("{}", &volume.published));
        metadata.add_metadata("volume", &volume.title);

        let mut output = File::create(&path)?;
        write!(output, "{}", format!("{}", metadata.build()))?;
    }

    Ok(())
}
