use regex::Regex;
use select::predicate::{Attr, Class};
use select::document::Document;
use select::node::Node;
use bytes::Bytes;
use reqwest::blocking::{ClientBuilder, Request};
use reqwest::Method;
use url::Url;
use thiserror::Error;
use chrono::NaiveDateTime;

#[derive(Error, Debug)]
pub enum Error {
    #[error("could not parse url")]
    UrlError(#[from] url::ParseError),
    #[error("request failed")]
    RequestError(#[from] reqwest::Error),
    #[error("unexpected response")]
    ResponseError,
    #[error("failed to parse response")]
    ParserError(std::io::Error),
    #[error("document selection failed")]
    SelectError(&'static str),
    #[error("image failed")]
    ImageError,
}

#[derive(Debug)]
pub struct RawVolume(String, Bytes);

const NAVER_POST_URL: &'static str = "https://post.naver.com/viewer/postView.nhn";
const USER_AGENT: &'static str = concat!("naver", "/", env!("CARGO_PKG_VERSION"));

impl RawVolume {
    pub fn get<T: Into<String> + AsRef<str>>(id: T) -> Result<Self, crate::Error> {
        let mut url = Url::parse(NAVER_POST_URL)?;
        url.query_pairs_mut().append_pair("volumeNo", id.as_ref());
        let client = ClientBuilder::new().user_agent(USER_AGENT).build()?;
        let resp = client.execute(Request::new(Method::GET, url))?;
        match resp.status().is_success() {
            true => Ok(RawVolume(id.into(), resp.bytes()?)),
            false => Err(Error::ResponseError)
        }
    }
}

pub struct VolumeParser(RawVolume);


impl VolumeParser {
    pub fn parse(raw: RawVolume) -> Result<Volume, crate::Error> {
        let document = Document::from_read(&*raw.1)
            .map_err(|err| crate::Error::ParserError(err))?;

        let title = Self::title(&document)?;
        let published = Self::published(&document)?;
        let series = Self::series(&document)?;
        let member = Self::member(&document)?;
        let images = Self::images(&document)?;

        Ok(Volume {
            id: raw.0.clone(),
            title,
            published,
            series,
            member,
            images,
        })
    }

    fn title(document: &Document) -> Result<String, crate::Error> {
        let predicate = Attr("property","nv:news:title");
        let selection = document.select(predicate);
        let node = selection.into_selection().first()
            .ok_or_else(|| crate::Error::SelectError("nv:news:title"))?;
        let value = node.attr("content")
            .ok_or_else(|| crate::Error::SelectError("nv:news:title content attribute"))?;
        Ok(value.to_string())
    }

    fn published(document: &Document) -> Result<i64, crate::Error> {
        let predicate = Attr("property","og:createdate");
        let selection = document.select(predicate);
        let node = selection.into_selection().first()
            .ok_or_else(|| crate::Error::SelectError("og:createdate"))?;
        let value = node.attr("content")
            .ok_or_else(|| crate::Error::SelectError("og:createdate content attribute"))?;
        let published = NaiveDateTime::parse_from_str(value, "%Y.%m.%d. %H:%M:%S")
            .map_err(|_| crate::Error::SelectError("og:createdate parsing"))?;
        Ok(published.timestamp())
    }

    fn series(document: &Document) -> Result<String, crate::Error> {
        let re = Regex::new(r"seriesNo=(\d+)&?").unwrap();
        let predicate = Attr("type","x-clip-content");
        let selection = document.select(predicate);
        let node = selection.into_selection().first()
            .ok_or_else(|| crate::Error::SelectError("x-clip-content"))?;
        let value = node.text();
        let captures = re.captures(&value)
            .ok_or_else(|| crate::Error::SelectError("seriesNo=(\\d+)&?"))?;
        Ok(captures.get(1)
            .ok_or_else(|| crate::Error::SelectError("seriesNo=(\\d+)&?"))?
            .as_str()
            .to_string())
    }

    fn member(document: &Document) -> Result<String, crate::Error> {
        let predicate = Attr("property","og:url");
        let selection = document.select(predicate);
        let node = selection.into_selection().first()
            .ok_or_else(|| crate::Error::SelectError("og:url"))?;
        let value = node.attr("content")
            .ok_or_else(|| crate::Error::SelectError("og:url content attribute"))?;
        let url = Url::parse(value)?;
        let pair = url.query_pairs().find(|item| item.0.eq("memberNo"))
            .ok_or_else(|| crate::Error::SelectError("og:url query param"))?;
        Ok(pair.1.to_string())
    }

    fn images(document: &Document) -> Result<Vec<Image>, crate::Error> {
        let predicate = Attr("type","x-clip-content");
        let selection = document.select(predicate);
        let node = selection.into_selection().first()
            .ok_or_else(|| crate::Error::SelectError("x-clip-content"))?;
        let value = node.text();
        Ok(Document::from(value.as_str())
            .select(Class("se_mediaImage"))
            .map(Image::new_from_volume_node)
            .filter_map(|image| image.ok())
            .collect())
    }
}

pub struct Volume {
    pub id: String,
    pub title: String,
    pub series: String,
    pub member: String,
    pub published: i64,
    pub images: Vec<Image>
}

impl Volume {
    pub fn get<T: Into<String> + AsRef<str>>(id: T) -> Result<Self, crate::Error> {
        Self::parse(RawVolume::get(id)?)
    }

    pub fn parse(raw: RawVolume) -> Result<Self, crate::Error> {
        VolumeParser::parse(raw)
    }
}

pub struct Image(String);

impl Image {
    pub fn new_from_volume_node(node: Node) -> Result<Self, crate::Error> {
        let src = node.attr("data-src").ok_or(crate::Error::ImageError)?;
        let mut url = Url::parse(src).map_err(|_| crate::Error::ImageError)?;
        url.set_query(None);
        Ok(Image(url.to_string()))
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn get_raw_volume() {
        let volume = RawVolume::get("31685068");
        dbg!(&volume);
        assert!(volume.is_ok());
    }

    #[test]
    fn parse_into_volume() {
        let volume = Volume::get("31685068").unwrap();
        assert_eq!(volume.id, "31685068");
        assert_eq!(volume.title, "[오마이걸] 귓가에 계속 들리는 그 멜로디♪ ‘Dun Dun Dance’ MV 촬영 현장!");
        assert_eq!(volume.published, 1623097830);
        assert_eq!(volume.series, "237429");
        assert_eq!(volume.member, "3717330");
        assert_eq!(volume.images.len(), 83);
        assert!(volume.images[0].0.ends_with("01.jpg"));
    }
}
