use nauty_pet::prelude::*;
use testing::{randomize_labels, GraphIter};

use criterion::{black_box, criterion_group, criterion_main, Criterion};
use petgraph::{
    algo::isomorphism::is_isomorphic,
    graph::{Graph, IndexType},
    Directed, Undirected,
};
use rand::distributions::Uniform;
use rand::prelude::*;
use rand_xoshiro::Xoshiro256Plus;

fn check_iso_un<Ix: IndexType>(
    graphs: &[(Graph<u8, u8, Undirected, Ix>, Graph<u8, u8, Undirected, Ix>)],
) -> bool {
    graphs.iter().all(|(g, h)| {
        g.clone().into_canon().is_identical(&h.clone().into_canon())
    })
}

fn check_iso_un_pet<Ix: IndexType>(
    graphs: &[(Graph<u8, u8, Undirected, Ix>, Graph<u8, u8, Undirected, Ix>)],
) -> bool {
    graphs.iter().all(|(g, h)| is_isomorphic(&g, &h))
}

fn check_iso_di<Ix: IndexType>(
    graphs: &[(Graph<u8, u8, Directed, Ix>, Graph<u8, u8, Directed, Ix>)],
) -> bool {
    graphs.iter().all(|(g, h)| {
        g.clone().into_canon().is_identical(&h.clone().into_canon())
    })
}

fn check_iso_di_pet<Ix: IndexType>(
    graphs: &[(Graph<u8, u8, Directed, Ix>, Graph<u8, u8, Directed, Ix>)],
) -> bool {
    graphs.iter().all(|(g, h)| is_isomorphic(&g, &h))
}

fn criterion_benchmark(c: &mut Criterion) {
    let mut rng = Xoshiro256Plus::seed_from_u64(0);
    let mut graphs = GraphIter::<Undirected>::default();
    graphs.edge_wt_distr = Uniform::from(0..=0);

    let graphs = Vec::from_iter(
        graphs
            .take(1000)
            .map(|g| (g.clone(), randomize_labels(g, &mut rng))),
    );
    c.bench_function("undirected", |b| {
        b.iter(|| check_iso_un(black_box(&graphs)))
    });
    c.bench_function("undirected (petgraph)", |b| {
        b.iter(|| check_iso_un_pet(black_box(&graphs)))
    });

    let mut graphs = GraphIter::<Directed>::default();
    graphs.edge_wt_distr = Uniform::from(0..=0);

    let graphs = Vec::from_iter(
        graphs
            .take(1000)
            .map(|g| (g.clone(), randomize_labels(g, &mut rng))),
    );
    c.bench_function("directed", |b| {
        b.iter(|| check_iso_di(black_box(&graphs)))
    });
    c.bench_function("directed (petgraph)", |b| {
        b.iter(|| check_iso_di_pet(black_box(&graphs)))
    });
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);
