use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::collections::HashMap;

pub type BasicResource = HashMap<String, Value>;

#[derive(Debug, Serialize, Deserialize)]
#[serde(untagged)]
pub enum Either<T, U> {
    Left(T),
    Right(U),
}

#[derive(Debug, Serialize, Deserialize)]
pub struct BatchRequest<T> {
    pub requests: Vec<T>,
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(tag = "status")]
pub enum RequestResult<T> {
    Ok { result: T },
    Error { message: String },
}

impl<T> RequestResult<T> {
    pub fn ok(result: T) -> Self {
        Self::Ok { result }
    }

    pub fn err(error: impl ToString) -> Self {
        Self::Error {
            message: error.to_string(),
        }
    }

    pub fn into_result(self) -> Result<T, String> {
        match self {
            Self::Ok { result } => Ok(result),
            Self::Error { message } => Err(message),
        }
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct GetResourcePayload {
    pub id: String,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct SetResourcePayload<R = BasicResource> {
    pub id: String,
    pub resource: Option<R>,
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(tag = "status")]
pub enum SetResourceResponse {
    Added,
    Updated,
    Removed,
    Ignored,
}

impl SetResourceResponse {
    /// Returns `true` if the set resource response is [`Added`].
    ///
    /// [`Added`]: SetResourceResponse::Added
    pub fn is_added(&self) -> bool {
        matches!(self, Self::Added)
    }

    /// Returns `true` if the set resource response is [`Updated`].
    ///
    /// [`Updated`]: SetResourceResponse::Updated
    pub fn is_updated(&self) -> bool {
        matches!(self, Self::Updated)
    }

    /// Returns `true` if the resource set response is [`Removed`].
    ///
    /// [`Removed`]: Response::Removed
    pub fn is_removed(&self) -> bool {
        matches!(self, Self::Removed)
    }

    /// Returns `true` if the set resource response is [`Ignored`].
    ///
    /// [`Ignored`]: SetResourceResponse::Ignored
    pub fn is_ignored(&self) -> bool {
        matches!(self, Self::Ignored)
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct TaggedResource<R = BasicResource> {
    pub id: String,
    pub resource: Option<R>,
}

#[cfg(feature = "async-graphql")]
#[async_graphql::Object]
impl<R> TaggedResource<R>
where
    R: async_graphql::OutputType,
{
    async fn id(&self) -> &str {
        &self.id
    }

    async fn resource(&self) -> Option<&R> {
        self.resource.as_ref()
    }
}
