use crate::{RequestResult, SetResourcePayload, SetResourceResponse};
use nats::asynk::Connection;
use serde::{de::DeserializeOwned, Serialize};
use std::io;

pub struct Resource<T>
where
    T: ResourceDefinition,
{
    nats: Connection,
    definition: T,
}

impl<T> Resource<T>
where
    T: ResourceDefinition + Default,
{
    pub fn new(nats: Connection) -> Self {
        Self {
            nats,
            definition: T::default(),
        }
    }
}

impl<T> Resource<T>
where
    T: ResourceDefinition,
{
    pub fn with_definition(nats: Connection, definition: T) -> Self {
        Self { nats, definition }
    }

    /// Set the resource with the given id. If the resource doesn't exist, it is added. If the
    /// resource does exist, it is updated. If the provided resource is None, and the resource
    /// exists on the server, it is removed. If the provided resource is None, and the resource
    /// doesn't exist on the server, the operation is ignored.
    ///
    /// You can inspect the return result to see what the server did as a result of this operation
    pub async fn set(
        &self,
        id: String,
        resource: Option<T::Resource>,
    ) -> io::Result<SetResourceResponse> {
        let msg = serde_json::to_vec(&SetResourcePayload { id, resource })?;
        let response = self
            .nats
            .request(&self.definition.set_subject(), &msg)
            .await?;
        let result = serde_json::from_slice::<RequestResult<SetResourceResponse>>(&response.data)?
            .into_result()
            .map_err(|message| io::Error::new(io::ErrorKind::Other, message))?;
        Ok(result)
    }
}

pub trait ResourceDefinition {
    type Resource: Serialize + DeserializeOwned;

    fn set_subject(&self) -> String;
}
