/*! This file describes the interface between an `Engine` and a user interface.*/
use std::collections::HashMap;

use anyhow::Result;
use serde::Deserialize;
/** The [EngineTrait] defines what an engine must be able to do */
pub trait EngineTrait {
    /// the configuration paramaeters for the engine, satisfying [ConfigTrait].
    type Config;
    /// the configuration paramaeters for the engine, satisfying [ActionTrait].
    type Action;
    /// the action sent to the engine, satisfying [ResponseTrait].
    type Response;
    /** 'new' creates an Engine from a `config` and an [InterfaceType] */
    fn new(config: &Self::Config, interface_type: InterfaceType) -> Result<Self>
    where
        Self: Sized;
    /** initial_html provides the initial HTML. */
    fn initial_html(&mut self) -> Result<String>;
    /** execute executes the user command (Action) and returns a Response. */
    fn execute(&mut self, action: Self::Action) -> Result<Self::Response>;
    /** `handle_event` handles an event generated by the app */
    fn handle_event(&mut self, event: &Event) -> Result<Self::Response>;
    /// interface type
    fn get_interface_type(&self) -> InterfaceType;
}
/** [ConfigTrait] is the requirements for the engine configuration (passed to the engine when it it created). */
pub trait ConfigTrait {
    fn from_json(json_str: &str) -> Result<Self>
    where
        Self: Sized;
}
/** [ActionTrait] is the requirements for an action (command sent from the user interface to the engine) */
pub trait ActionTrait {
    fn from_json(json_str: &str) -> Result<Self>
    where
        Self: Sized;
}
/** [ResponseTrait] is the requirements for a response (sent from the engine to the user interface). One possible response is that shutdown is required */
pub trait ResponseTrait {
    fn shutdown_required(&self) -> bool;
}
/** [Event]s are requirements for an event (generated by the execution environment and passed to the engine). Events are like actions but generated by the app not the user interface. */
#[derive(Debug, Deserialize, Clone)]
//#[repr(C)]
pub enum Event {
    /// at start, after the user interface is ready
    Create,
    /// asks for a summary of the engine state so that the app can be restarted
    SaveInstanceState,
    /// asks to set the engine state
    RestoreInstanceState(HashMap<String, String>),
    /// notify the engine that the app is being shut down
    Stop,
}
impl Event {
    pub fn from_json(json_str: &str) -> Result<Self>
    where
        Self: Sized,
    {
        Ok(serde_json::from_str(json_str)?)
    }
}
#[derive(Debug, Clone, Copy, strum::Display)]
/** type of execution environment e.g. PC, Android */
#[repr(C)]
pub enum InterfaceType {
    PC,
    Android,
}
impl Default for InterfaceType {
    fn default() -> Self {
        Self::PC
    }
}

/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */
