// You can run this example on different databases (Docker files provided in
// the repository root).
// Select the database by compiling this example with the corresponding feature:
// * sqlite
// * mysql
// Also make sure that the correct type DbBackend is uncommented, see below.

#[macro_use]
extern crate diesel;

use chrono::prelude::NaiveDateTime;
use naphtha::{
    barrel::{
        types,
        DatabaseSqlMigration,
        DatabaseSqlMigrationExecutor,
        Migration,
    },
    model,
    DatabaseConnect,
    DatabaseConnection,
    DatabaseInsertHandler,
    DatabaseModel,
    DatabaseModelModifier,
    DatabaseRemoveHandler,
    DatabaseUpdateHandler,
};

const DATABASE_URL: &'static str = if cfg!(feature = "sqlite") {
    ":memory:"
} else if cfg!(feature = "mysql") {
    "mysql://naphtha:naphtha@127.0.0.1:3306/naphtha"
} else {
    "not supported"
};

//
//type DbBackend = diesel::SqliteConnection;
type DbBackend = diesel::MysqlConnection;

// The model attribute automatically adds:
//
// use schema::*;
// #[cfg(any(feature = "sqlite", feature = "mysql"))]
// use diesel::{backend::Backend, prelude::*};
// #[derive(Debug, Queryable, Identifiable, AsChangeset, Associations)]
// #[table_name = "persons"]
#[model(table_name = "persons")]
pub struct Person {
    id: i32,
    pub description: Option<String>,
    pub updated_at: NaiveDateTime,
}

pub mod schema {
    table! {
        persons (id) {
            id -> Int4,
            description -> Nullable<Varchar>,
            updated_at -> Timestamp,
        }
    }
}

impl naphtha::DatabaseModel for Person {
    type PrimaryKey = i32;
    fn primary_key(&self) -> Self::PrimaryKey {
        self.id
    }

    fn set_primary_key(&mut self, value: &Self::PrimaryKey) {
        self.id = *value;
    }

    fn default_primary_key() -> Self::PrimaryKey {
        0
    }

    fn table_name() -> &'static str {
        "persons"
    }
}

impl<T> DatabaseUpdateHandler<T> for Person {
    fn pre_update(&mut self, _conn: &DatabaseConnection<T>) {
        self.updated_at = chrono::Utc::now().naive_utc();
    }

    fn post_update(&mut self, _conn: &DatabaseConnection<T>) {}
}

impl<T> DatabaseInsertHandler<T> for Person {}
impl<T> DatabaseRemoveHandler<T> for Person {}

// The following trait is generated by naphtha-proc-macro
//
//
///// Queries the model by the given property. Returns only those with an
///// exact match.
//pub trait QueryByProperties<DB>
//where
//    Self: Sized,
//{
//    type Error;
//    /// Queries the database with by the given property. It only returns
//    /// those with an exact match.
//    fn query_by_id(conn: &DatabaseConnection<DB>, property: &i32) -> Result<Self, Self::Error>;
//    /// Queries the database with by the given property. It only returns
//    /// those with an exact match.
//    fn query_by_description(
//        conn: &DatabaseConnection<DB>,
//        property: &Option<String>,
//    ) -> Result<Vec<Self>, Self::Error>;
//    fn query_by_ids(conn: &DatabaseConnection<DB>, ids: &[i32]) -> Result<Vec<Self>, Self::Error>;
//}
//
// The following implementation is generated by naphtha-proc-macro.
// This implementation depends on the build. The following example is
// generated for SqliteConnection.
//
//impl QueryByProperties<SqliteConnection> for Person {
//    type Error = ::diesel::result::Error;
//    fn query_by_id(
//        conn: &DatabaseConnection<SqliteConnection>,
//        property: &i32,
//    ) -> ::diesel::result::QueryResult<Self> {
//        use schema::{persons, persons::dsl::*};
//        conn.custom::<::diesel::result::QueryResult<Self>, _>(|c| {
//            persons.filter(id.eq(property)).first::<Self>(&*c)
//        })
//    }
//    fn query_by_description(
//        conn: &DatabaseConnection<SqliteConnection>,
//        property: &Option<String>,
//    ) -> ::diesel::result::QueryResult<Vec<Self>> {
//        use schema::{persons, persons::dsl::*};
//        conn.custom::<::diesel::result::QueryResult<Vec<Self>>, _>(|c| {
//            persons.filter(description.eq(property)).load::<Self>(&*c)
//        })
//    }
//    fn query_by_ids(
//        conn: &DatabaseConnection<SqliteConnection>,
//        ids: &[i32],
//    ) -> ::diesel::result::QueryResult<Vec<Self>> {
//        use {
//            schema::{persons, persons::dsl::*},
//        };
//        conn.custom::<::diesel::result::QueryResult<Vec<Self>>, _>(|c| {
//            persons.filter(id.eq_any(ids)).load::<Self>(&*c)
//        })
//    }
//}

#[cfg(any(feature = "barrel-sqlite", feature = "barrel-mysql",))]
impl DatabaseSqlMigration for Person {
    fn migration_up(migration: &mut Migration) {
        migration.create_table_if_not_exists(Self::table_name(), |t| {
            t.add_column("id", types::primary());
            t.add_column("description", types::text().nullable(true));
            t.add_column("updated_at", types::custom("timestamp"));
        });
    }

    fn migration_down(migration: &mut Migration) {
        migration.drop_table_if_exists(Self::table_name());
    }
}

fn main() {
    println!("Using {}", DATABASE_URL);
    let db = DatabaseConnection::connect(DATABASE_URL).unwrap();

    // create the table if not existent
    // This method can be used on startup of your application to make sure
    // your database schema is always up to date.
    match Person::execute_migration_up(&db) {
        Ok(_) => (),
        Err(msg) => println!("Could not create table: {}", msg),
    };

    let mut p = Person {
        id: Person::default_primary_key(),
        description: Some("The new person is registered".into()),
        updated_at: chrono::Utc::now().naive_utc(),
    };

    p.insert(&db);
    // id member is set to the correct number given by the database.

    // do a custom query to the database
    let res =
        db.custom::<diesel::result::QueryResult<Person>, _>(|c: &DbBackend| {
            use schema::persons::dsl::*;
            persons.filter(id.eq(1)).first(c)
        });
    let queried_by_id = Person::query_by_id(&db, &1);
    println!("{:#?}", res);
    println!("{:#?}", queried_by_id);

    p.remove(&db);
    // p not available anymore

    match Person::execute_migration_down(&db) {
        Ok(_) => (),
        Err(msg) => println!("Could not drop table: {}", msg),
    };
}
