#[macro_use]
extern crate diesel;

use chrono::prelude::NaiveDateTime;
use naphtha::{
    barrel::{types, DatabaseSqlMigration, Migration},
    model,
    DatabaseInsertHandler,
    DatabaseRemoveHandler,
    DatabaseUpdateHandler,
};

// The model attribute automatically adds:
//
// use schema::*;
// #[derive(Debug, Queryable, Identifiable, AsChangeset, Associations)]
// #[table_name = "persons"]
#[model(table_name = "persons")]
pub struct Person {
    id: i32,
    pub description: Option<String>,
    pub updated_at: NaiveDateTime,
}

pub mod schema {
    table! {
        persons (id) {
            id -> Int4,
            description -> Nullable<Varchar>,
            updated_at -> Timestamp,
        }
    }
}

impl naphtha::DatabaseModel for Person {
    type PrimaryKey = i32;
    fn primary_key(&self) -> Self::PrimaryKey {
        self.id
    }

    fn set_primary_key(&mut self, value: &Self::PrimaryKey) {
        self.id = *value;
    }

    fn default_primary_key() -> Self::PrimaryKey {
        0
    }

    fn table_name() -> &'static str {
        "persons"
    }
}

impl<T> DatabaseUpdateHandler<T> for Person {
    fn pre_update(&mut self, _conn: &DatabaseConnection<T>) {
        self.updated_at = chrono::Utc::now().naive_utc();
    }

    fn post_update(&mut self, _conn: &DatabaseConnection<T>) {}
}

impl<T> DatabaseInsertHandler<T> for Person {}
impl<T> DatabaseRemoveHandler<T> for Person {}

// The following trait is generated by naphtha-proc-macro
//
//
///// Queries the model by the given property. Returns only those with an
///// exact match.
//pub trait QueryByProperties<DB>
//where
//    Self: Sized,
//{
//    type Error;
//    /// Queries the database with by the given property. It only returns
//    /// those with an exact match.
//    fn query_by_id(conn: &DatabaseConnection<DB>, property: &i32) -> Result<Self, Self::Error>;
//    /// Queries the database with by the given property. It only returns
//    /// those with an exact match.
//    fn query_by_description(
//        conn: &DatabaseConnection<DB>,
//        property: &Option<String>,
//    ) -> Result<Vec<Self>, Self::Error>;
//    fn query_by_ids(conn: &DatabaseConnection<DB>, ids: &[i32]) -> Result<Vec<Self>, Self::Error>;
//}
//
// The following implementation is generated by naphtha-proc-macro.
// This implementation depends on the build. The following example is
// generated for SqliteConnection.
//
//impl QueryByProperties<SqliteConnection> for Person {
//    type Error = ::diesel::result::Error;
//    fn query_by_id(
//        conn: &DatabaseConnection<SqliteConnection>,
//        property: &i32,
//    ) -> ::diesel::result::QueryResult<Self> {
//        use schema::{persons, persons::dsl::*};
//        conn.custom::<::diesel::result::QueryResult<Self>, _>(|c| {
//            persons.filter(id.eq(property)).first::<Self>(&*c)
//        })
//    }
//    fn query_by_description(
//        conn: &DatabaseConnection<SqliteConnection>,
//        property: &Option<String>,
//    ) -> ::diesel::result::QueryResult<Vec<Self>> {
//        use schema::{persons, persons::dsl::*};
//        conn.custom::<::diesel::result::QueryResult<Vec<Self>>, _>(|c| {
//            persons.filter(description.eq(property)).load::<Self>(&*c)
//        })
//    }
//    fn query_by_ids(
//        conn: &DatabaseConnection<SqliteConnection>,
//        ids: &[i32],
//    ) -> ::diesel::result::QueryResult<Vec<Self>> {
//        use {
//            schema::{persons, persons::dsl::*},
//        };
//        conn.custom::<::diesel::result::QueryResult<Vec<Self>>, _>(|c| {
//            persons.filter(id.eq_any(ids)).load::<Self>(&*c)
//        })
//    }
//}

#[cfg(any(feature = "barrel-full", feature = "barrel-sqlite",))]
impl DatabaseSqlMigration for Person {
    fn migration_up(migration: &mut Migration) {
        use naphtha::DatabaseModel;
        migration.create_table_if_not_exists(Self::table_name(), |t| {
            t.add_column("id", types::primary());
            t.add_column("description", types::text().nullable(true));
            t.add_column("updated_at", types::custom("timestamp"));
        });
    }

    fn migration_down(migration: &mut Migration) {
        use naphtha::DatabaseModel;
        migration.drop_table_if_exists(Self::table_name());
    }
}

fn main() {
    use naphtha::{
        barrel::DatabaseSqlMigrationExecutor,
        DatabaseConnect,
        DatabaseModel,
    };

    let db = DatabaseConnection::connect(":memory:").unwrap();

    // create the table if not existent
    // This method can be used on startup of your application to make sure
    // your database schema is always up to date.
    match Person::execute_migration_up(&db) {
        Ok(_) => (),
        Err(msg) => println!("Could not create table: {}", msg),
    };

    let mut p = Person {
        id: Person::default_primary_key(),
        description: Some("The new person is registered".into()),
        updated_at: chrono::Utc::now().naive_utc(),
    };

    p.insert(&db);
    // id member is set to the correct number given by the database.

    // do a custom query to the database
    let res = db.custom::<diesel::result::QueryResult<Person>, _>(|c| {
        use schema::persons::dsl::*;
        persons.filter(id.eq(1)).first(c)
    });
    let queried_by_id = Person::query_by_id(&db, &1);
    println!("{:#?}", res);
    println!("{:#?}", queried_by_id);

    p.remove(&db);
    // p not available anymore
}
