// SPDX-License-Identifier: MPL-2.0

use std::fs::OpenOptions;
use std::io::Write;
use std::os::unix::fs::OpenOptionsExt;
use std::path::PathBuf;

use anyhow::{Context, Result};
use nix::sys::stat;
use nix::unistd;

use crate::random;
use crate::random::CharSet;

pub struct TempFile {
    pub path: PathBuf,
    dirpath: PathBuf,
}

impl TempFile {
    pub fn new() -> TempFile {
        let tempdir = "/dev/shm";
        let nchars = 16;

        let dir_random = random::random_string(CharSet::Alnum, nchars);
        let file_random = random::random_string(CharSet::Alnum, nchars);

        let dirpath = format!("{}/napa-{}", tempdir, dir_random);
        let filepath = format!("{}/{}.txt", dirpath, file_random);

        TempFile {
            path: PathBuf::from(filepath),
            dirpath: PathBuf::from(dirpath),
        }
    }

    pub fn write(&self, content: &str) -> Result<()> {
        // Unfortunately the stdlib doesn't allow setting the permissions when creating a
        // directory, so unix it is.
        unistd::mkdir(&self.dirpath, stat::Mode::S_IRWXU)
            .with_context(|| format!("Unable to create directory {}", self.dirpath.display()))?;

        // There is also fs::write, but we need to explicitly state
        // the permissions when creating this file.
        let mut file = OpenOptions::new()
            .write(true)
            .create_new(true)
            .mode(0o600) // u+rw
            .open(&self.path)
            .with_context(|| format!("Unable to create file {}", self.path.display()))?;

        file.write_all(content.as_bytes())
            .with_context(|| format!("Unable to write to file {}", self.path.display()))?;

        // Helpful to check for file system errors
        file.sync_all()
            .with_context(|| format!("Unable to synchronise file {}", self.path.display()))
    }

    pub fn read(&self) -> Result<String> {
        std::fs::read_to_string(&self.path).with_context(|| format!("Unable to read from file {}", self.path.display()))
    }

    pub fn remove(self) -> Result<()> {
        std::fs::remove_dir_all(&self.dirpath)
            .with_context(|| format!("Unable to remove directory {}", self.dirpath.display()))?;
        // Consume this value so the destructor isn't run after
        std::mem::forget(self);
        Ok(())
    }
}

impl Drop for TempFile {
    fn drop(&mut self) {
        // Ignore the return value since we shouldn't panic in a destructor
        let _ = std::fs::remove_dir_all(&self.dirpath);
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn content_written_to_tempfile_can_be_read() {
        let file = TempFile::new();

        let contents = "Hello world!";
        file.write("Hello world!").unwrap();
        assert_eq!(file.read().unwrap(), contents);
    }

    #[test]
    fn tempfile_is_deleted_on_removal() {
        let file = TempFile::new();
        let dirpath = file.dirpath.clone();

        file.write("").unwrap();
        file.remove().unwrap();

        assert!(!dirpath.exists())
    }

    #[test]
    fn tempfile_is_deleted_on_drop() {
        let file = TempFile::new();
        let dirpath = file.dirpath.clone();

        file.write("").unwrap();
        drop(file);

        assert!(!dirpath.exists())
    }
}
