// SPDX-License-Identifier: MPL-2.0

use std::io::BufRead;
use std::path::Path;

use anyhow::Result;

use crate::crypto::chacha20::Key;
use crate::entries::Entries;
use crate::random;
use crate::random::CharSet;
use crate::repl::util;

const PASSWORD_PREFIX: &str = "password:";

pub fn find_password(content: &str) -> Option<String> {
    for line in content.lines() {
        if let Some(stripped) = line.strip_prefix(PASSWORD_PREFIX) {
            let password = stripped.trim().to_string();
            return Some(password);
        }
    }

    None
}

fn replace_password(content: &str, password: &str) -> String {
    let mut new_content = String::new();

    let mut found_pass = false;
    for line in content.lines() {
        let new_line = if line.starts_with(PASSWORD_PREFIX) && !found_pass {
            found_pass = true;
            format!("{} {}", PASSWORD_PREFIX, password)
        } else {
            line.to_string()
        };
        new_content.push_str(&new_line);
        new_content.push('\n');
    }

    new_content.trim().to_string()
}

pub fn gen(name: &str, length: u32, charset: CharSet, database_path: &Path, key: &Key) -> Result<()> {
    let (mut header, mut entries) = util::read_database(database_path, key)?;
    let new_password = random::random_string(charset, length);
    gen_impl(name, &new_password, &mut entries, &mut std::io::stdin().lock());
    util::write_database(database_path, &mut header, &entries, key)
}

fn gen_impl(name: &str, new_password: &str, entries: &mut Entries, input: &mut impl BufRead) {
    if let Some(content) = entries.get_mut(name) {
        if find_password(content).is_some() {
            let prompt = format!("Password for {} already exists. Overwrite it? [y/N] ", name);
            let overwrite = util::confirm_prompt(&prompt, input);
            if overwrite {
                *content = replace_password(content, new_password);
                println!("Password for {} overwritten.", name);
            } else {
                println!("Password for {} not overwritten.", name);
            }
        } else {
            // Else no password line found, so append the new password to the end of the entry
            if !content.is_empty() {
                content.push('\n');
            }

            content.push_str(PASSWORD_PREFIX);
            content.push(' ');
            content.push_str(new_password);
            println!("New password generated for {}.", name);
        }
    } else {
        // No existing entry, create new one
        let content = format!("{} {}", PASSWORD_PREFIX, new_password);
        entries.insert(name.to_string(), content);
        println!("New password generated for {}.", name);
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn find_password_returns_none_if_no_password() {
        let content = "no password here";
        assert!(find_password(content).is_none());
    }

    #[test]
    fn find_password_returns_first_password() {
        let content = "first line\npassword: first password  \npassword: second password";
        assert_eq!(find_password(content), Some("first password".to_string()));
    }

    #[test]
    fn replace_password_does_nothing_if_no_password_found() {
        let content = "no password here";
        assert_eq!(replace_password(content, "new password"), content);
    }

    #[test]
    fn replace_password_replaces_first_password() {
        let content = "first line\npassword: first password   \npassword: second password";
        assert_eq!(
            replace_password(content, "new password"),
            "first line\npassword: new password\npassword: second password"
        );
    }

    #[test]
    fn gen_overwrites_password_if_confirmed() {
        let mut entries = Entries::new();

        let entry = "Entry";
        let new_password = "new_password";
        let old_value = "some stuff\npassword: old_password\nother stuff";
        let new_value = "some stuff\npassword: new_password\nother stuff";

        entries.insert(entry.to_string(), old_value.to_string());

        let response = "y\n";
        gen_impl(entry, new_password, &mut entries, &mut response.as_bytes());

        assert_eq!(entries.get(entry).unwrap(), new_value);
    }

    #[test]
    fn gen_does_not_overwrite_password_if_not_confirmed() {
        let mut entries = Entries::new();

        let entry = "Entry";
        let new_password = "new_password";
        let old_value = "some stuff\npassword: old_password\nother stuff";

        entries.insert(entry.to_string(), old_value.to_string());

        let response = "n\n";
        gen_impl(entry, new_password, &mut entries, &mut response.as_bytes());

        assert_eq!(entries.get(entry).unwrap(), old_value);
    }

    #[test]
    fn gen_adds_password_to_end_if_old_password_does_not_exist() {
        let mut entries = Entries::new();

        let entry = "Entry";
        let new_password = "secret!";
        let old_value = "some stuff";
        let new_value = "some stuff\npassword: secret!";

        entries.insert(entry.to_string(), old_value.to_string());

        gen_impl(entry, new_password, &mut entries, &mut "".as_bytes());

        assert_eq!(entries.get(entry).unwrap(), new_value);
    }

    #[test]
    fn gen_creates_new_entry_if_doesnt_exist() {
        let mut entries = Entries::new();

        let entry = "Entry";
        let new_password = "secret!";

        gen_impl(entry, new_password, &mut entries, &mut "".as_bytes());

        assert_eq!(entries.get(entry).unwrap(), "password: secret!");
    }
}
