// SPDX-License-Identifier: MPL-2.0

use std::path::Path;
use std::process::Command;

use anyhow::{bail, Context, Result};

use crate::crypto::chacha20::Key;
use crate::entries::Entries;
use crate::repl::gen;
use crate::repl::util;

pub fn clip(name: &str, database_path: &Path, key: &Key) -> Result<()> {
    let (_, entries) = util::read_database(database_path, key)?;
    clip_impl(name, &entries)?;
    Ok(())
}

fn clip_impl(name: &str, entries: &Entries) -> Result<()> {
    if let Some(content) = entries.get(name) {
        if let Some(password) = gen::find_password(content) {
            copy_to_clipboard(password)?;
            println!("Password for {} copied to clipboard.", name);
        } else {
            println!("Error: no password for {} found.", name);
        }
    } else {
        println!("Error: no entry for {} found.", name);
    }
    Ok(())
}

fn copy_to_clipboard(password: String) -> Result<()> {
    let status = Command::new("wl-copy")
        .arg("--paste-once")
        .arg("--type=text/plain")
        .arg(password)
        .status()
        .context("Unable to execute wl-copy")?;

    if !status.success() {
        bail!("Execution of wl-copy failed");
    }

    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;

    const ENTRY: &str = "Entry";
    const VALUE: &str = "password: hello!";

    fn paste() -> String {
        let output = Command::new("wl-paste")
            .arg("--type=text/plain")
            .arg("--no-newline")
            .output()
            .unwrap();

        String::from_utf8(output.stdout).unwrap()
    }

    #[test]
    #[ignore] // Doesn't work in CI environment since wayland isn't running
    fn password_of_entry_is_copied_to_clipboard() {
        let mut entries = Entries::new();
        entries.insert(ENTRY.to_string(), VALUE.to_string());

        clip_impl(ENTRY, &entries).unwrap();

        let pasted = paste();
        assert_eq!("hello!", pasted);
    }
}
