// SPDX-License-Identifier: MPL-2.0

use clap::ArgEnum;
use rand::rngs::OsRng;
use rand::seq::SliceRandom;

#[derive(Clone, Copy, Debug, ArgEnum)]
pub enum CharSet {
    Digit,
    Lower,
    Upper,
    Punct,
    Alpha,
    Alnum,
    Graph,
}

pub const DIGIT: &str = "0123456789";
pub const LOWER: &str = "abcdefghijklmnopqrstuvwxyz";
pub const UPPER: &str = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
pub const PUNCT: &str = "!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~";

// Hmm, how to reduce this duplication?
pub const ALPHA: &str = concat!("abcdefghijklmnopqrstuvwxyz", "ABCDEFGHIJKLMNOPQRSTUVWXYZ",);

pub const ALNUM: &str = concat!("abcdefghijklmnopqrstuvwxyz", "ABCDEFGHIJKLMNOPQRSTUVWXYZ", "0123456789",);

pub const GRAPH: &str = concat!(
    "abcdefghijklmnopqrstuvwxyz",
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ",
    "0123456789",
    "!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~",
);

impl CharSet {
    fn chars(self) -> Vec<char> {
        match self {
            CharSet::Digit => DIGIT,
            CharSet::Lower => LOWER,
            CharSet::Upper => UPPER,
            CharSet::Punct => PUNCT,
            CharSet::Alpha => ALPHA,
            CharSet::Alnum => ALNUM,
            CharSet::Graph => GRAPH,
        }
        .chars()
        .collect()
    }
}

pub fn random_string(charset: CharSet, length: u32) -> String {
    let chars = charset.chars();
    (0..length)
        .map(|_| chars.choose(&mut OsRng).unwrap())
        .collect::<String>()
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn random_string_length_matches_input_length() {
        let string = random_string(CharSet::Graph, 20);
        assert_eq!(string.len(), 20);
    }

    #[test]
    fn random_string_characters_are_taken_from_charset() {
        let charset = CharSet::Graph;
        let string = random_string(charset, 20);
        assert!(string.chars().all(|c| charset.chars().contains(&c)))
    }
}
