// SPDX-License-Identifier: MPL-2.0

// Bindings and tests in this file based on sodiumoxide:
// https://github.com/sodiumoxide/sodiumoxide/blob/master/src/crypto/aead/xchacha20poly1305_ietf.rs

use anyhow::{bail, Result};
use libsodium_sys as ffi;
use nix::libc::c_ulonglong;
use rand::rngs::OsRng;
use rand::RngCore;

const TAGBYTES: usize = ffi::crypto_aead_xchacha20poly1305_ietf_ABYTES as usize;

pub struct Key(pub [u8; Key::SIZE]);

#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub struct Nonce(pub [u8; Nonce::SIZE]);

impl Key {
    pub const SIZE: usize = 32;
}

impl Nonce {
    pub const SIZE: usize = 24;

    pub fn generate() -> Nonce {
        let mut buf = [0u8; Nonce::SIZE];
        OsRng.fill_bytes(&mut buf);
        Nonce(buf)
    }
}

impl Default for Nonce {
    fn default() -> Nonce {
        Nonce([0u8; Nonce::SIZE])
    }
}

pub fn encrypt(message: &[u8], data: &[u8], nonce: &Nonce, key: &Key) -> Result<Vec<u8>> {
    let mut ciphertext = Vec::with_capacity(message.len() + TAGBYTES);
    let mut ciphertext_len: c_ulonglong = ciphertext.len().try_into().unwrap();

    unsafe {
        let ret = ffi::crypto_aead_xchacha20poly1305_ietf_encrypt(
            ciphertext.as_mut_ptr(),
            &mut ciphertext_len,
            message.as_ptr(),
            message.len().try_into().unwrap(),
            data.as_ptr(),
            data.len().try_into().unwrap(),
            std::ptr::null_mut(),
            nonce.0.as_ptr(),
            key.0.as_ptr(),
        );
        if ret != 0 {
            bail!("Encryption failed");
        }
        ciphertext.set_len(ciphertext_len.try_into().unwrap());
    }

    Ok(ciphertext)
}

pub fn decrypt(ciphertext: &[u8], data: &[u8], nonce: &Nonce, key: &Key) -> Result<Vec<u8>> {
    if ciphertext.len() < TAGBYTES {
        bail!(
            "Ciphertext length {} is less than tag length {}",
            ciphertext.len(),
            TAGBYTES
        );
    }

    let mut message = Vec::with_capacity(ciphertext.len() - TAGBYTES);
    let mut message_len: c_ulonglong = message.len().try_into().unwrap();

    unsafe {
        let ret = ffi::crypto_aead_xchacha20poly1305_ietf_decrypt(
            message.as_mut_ptr(),
            &mut message_len,
            std::ptr::null_mut(),
            ciphertext.as_ptr(),
            ciphertext.len().try_into().unwrap(),
            data.as_ptr(),
            data.len().try_into().unwrap(),
            nonce.0.as_ptr(),
            key.0.as_ptr(),
        );
        if ret != 0 {
            bail!("Decryption failed");
        }
        message.set_len(message_len.try_into().unwrap());
    };

    Ok(message)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn key_size_matches() {
        assert_eq!(Key::SIZE, ffi::crypto_aead_xchacha20poly1305_ietf_KEYBYTES as usize);
    }

    #[test]
    fn nonce_size_matches() {
        assert_eq!(Nonce::SIZE, ffi::crypto_aead_xchacha20poly1305_ietf_NPUBBYTES as usize);
    }

    #[test]
    fn test_vector() {
        let message = vec![
            0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c, 0x65, 0x6d,
            0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x6f, 0x66,
            0x20, 0x27, 0x39, 0x39, 0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f,
            0x66, 0x66, 0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20,
            0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75, 0x72, 0x65,
            0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73, 0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20,
            0x62, 0x65, 0x20, 0x69, 0x74, 0x2e,
        ];
        let data = vec![0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7];
        let key = Key([
            0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91,
            0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
        ]);
        let nonce = Nonce([
            0xf2, 0x8a, 0x50, 0xa7, 0x8a, 0x7e, 0x23, 0xc9, 0xcb, 0xa6, 0x78, 0x34, 0x66, 0xf8, 0x03, 0x59, 0x0f, 0x04,
            0xe9, 0x22, 0x31, 0xa3, 0x2d, 0x5d,
        ]);
        let ciphertext = vec![
            0x20, 0xf1, 0xae, 0x75, 0xe1, 0xe5, 0xe0, 0x00, 0x40, 0x29, 0x4f, 0x0f, 0xb1, 0x0e, 0xbb, 0x08, 0x10, 0xc5,
            0x93, 0xc7, 0xdb, 0xa4, 0xec, 0x10, 0x4c, 0x1e, 0x5e, 0xf9, 0x50, 0x7f, 0xae, 0xef, 0x58, 0xfc, 0x28, 0x98,
            0xbb, 0xd0, 0xe4, 0x7b, 0x2f, 0x53, 0x31, 0xfb, 0xc3, 0x67, 0xd3, 0xc2, 0x78, 0x4e, 0x36, 0x48, 0xce, 0x1e,
            0xaa, 0x77, 0x87, 0xad, 0x18, 0x6d, 0xb2, 0x68, 0x5e, 0xe8, 0x9a, 0xe4, 0xd3, 0x44, 0x1f, 0x6e, 0xa0, 0xb2,
            0x22, 0x4c, 0xd5, 0xa1, 0x34, 0x16, 0x1b, 0x55, 0x4d, 0x8b, 0x48, 0x35, 0x0b, 0x4a, 0xd4, 0x01, 0x15, 0xdb,
            0x81, 0xea, 0x82, 0x09, 0x68, 0xe9, 0x43, 0x89, 0x2f, 0x2b, 0x80, 0x51, 0xcb, 0x5f, 0x7a, 0x86, 0x66, 0xe7,
            0xe7, 0xef, 0x7f, 0x84, 0xc0, 0xa2, 0xf8, 0x0a, 0x12, 0xd0, 0x66, 0x80, 0xc8, 0xee, 0xbb, 0xd9, 0x30, 0x04,
            0x10, 0x9d, 0xe8, 0x42,
        ];

        let actual_ciphertext = encrypt(&message, &data, &nonce, &key).unwrap();
        assert_eq!(ciphertext, actual_ciphertext);

        let actual_message = decrypt(&ciphertext, &data, &nonce, &key).unwrap();
        assert_eq!(message, actual_message);
    }
}
