// SPDX-License-Identifier: MPL-2.0

use std::path::Path;
use std::pin::Pin;

use anyhow::Result;

use crate::cli::key;
use crate::cli::passphrase;
use crate::crypto::chacha20::Key;
use crate::database::Database;
use crate::entries;
use crate::header::Header;
use crate::repl;

pub fn open(database_path: &Path) -> Result<()> {
    let key = read_key(database_path)?;
    repl::repl(database_path, &key)
}

fn read_key(database_path: &Path) -> Result<Pin<Box<Key>>> {
    let database = Database::read(database_path)?;
    let header = Header::deserialize(&database.header)?;

    let passphrase = passphrase::read_passphrase("Enter the passphrase: ");
    let key = key::derive_key(&passphrase, &header)?;

    // Attempt to decrypt the database now. If it fails, then you (probably) entered the wrong passphrase.
    // Once the key is loaded into memory, further decryption failures indicate the database has been
    // corrupted or tampered with.
    entries::decrypt_soft(&database, &header, &key)?;

    Ok(key)
}
