// SPDX-License-Identifier: MPL-2.0

mod clear;
mod clip;
mod del;
mod do_move;
mod edit;
mod gen;
mod list;
mod show;
mod util;

use std::io::Write;
use std::path::Path;

use anyhow::Result;
use rustyline::Editor;
use rustyline::error::ReadlineError;
use structopt::StructOpt;
use structopt::clap::AppSettings;

use crate::crypto::chacha20::Key;
use crate::random::CharSet;

#[derive(Debug, StructOpt)]
#[structopt(setting(AppSettings::NoBinaryName))]
pub enum Repl {
    #[structopt(about = "List the existing entries in the database")]
    List,
    #[structopt(about = "Edit the entry using $EDITOR")]
    Edit {
        entry_name: String
    },
    #[structopt(about = "Print the contents of the entry")]
    Show {
        entry_name: String
    },
    #[structopt(about = "Delete the entry")]
    Del {
        entry_name: String
    },
    #[structopt(about = "Rename the entry")]
    Move {
        old_name: String,
        new_name: String,
    },
    #[structopt(about = "Generate a new password for the entry")]
    Gen {
        entry_name: String,
        #[structopt(short, long, default_value = "25")]
        length: u32,
        #[structopt(short, long, possible_values = &CharSet::variants(), default_value = "alnum")]
        charset: CharSet,
    },
    #[structopt(about = "Copy the entry password to the clipboard")]
    Clip {
        entry_name: String,
    },
    #[structopt(about = "Clear the screen")]
    Clear,
    #[structopt(about = "Close the database")]
    Exit,
}

impl Repl {
    fn run(&self, database_path: &Path, key: &Key) -> Result<bool> {
        let mut exit = false;

        match self {
            Repl::List => list::list(database_path, key)?,

            Repl::Edit { entry_name } => edit::edit(entry_name, database_path, key)?,
            Repl::Show { entry_name } => show::show(entry_name, database_path, key)?,
            Repl::Del { entry_name } => del::del(entry_name, database_path, key)?,
            Repl::Move { old_name, new_name } => do_move::do_move(old_name, new_name, database_path, key)?,

            Repl::Clear => clear::clear(),

            Repl::Gen { entry_name, length, charset } => gen::gen(entry_name, *length, *charset, database_path, key)?,
            Repl::Clip { entry_name } => clip::clip(entry_name, database_path, key)?,

            Repl::Exit => exit = true,
        }

        Ok(exit)
    }
}

pub fn repl(database_path: &Path, key: &Key) -> Result<()> {
    let mut rustyline = Editor::<()>::new();

    let mut exit = false;

    while !exit {
        let readline = rustyline.readline("> ");
        exit = match readline {
            Ok(line) => process_line(database_path, key, &line)?,
            Err(ReadlineError::Interrupted) => true, // CTRL-C
            Err(ReadlineError::Eof) => true,         // CTRL-D
            Err(err) => {
                println!("Error: {:?}", err);
                true
            }
        };
        std::io::stdout().flush().unwrap();
    }

    Ok(())
}

fn process_line(database_path: &Path, key: &Key, line: &str) -> Result<bool> {
    let commands = line
        .split_whitespace()
        .map(String::from)
        .collect::<Vec<String>>();

    if commands.is_empty() {
        return Ok(false);
    }

    let result = Repl::from_iter_safe(commands);

    match result {
        Ok(args) => args.run(database_path, key),
        Err(error) => { println!("{}", error.message); Ok(false) }
    }
}
