// SPDX-License-Identifier: MPL-2.0

use std::path::Path;

use anyhow::Result;
use wl_clipboard_rs::copy::{MimeType, Options, ServeRequests, Source};

use crate::crypto::chacha20::Key;
use crate::entries::Entries;
use crate::repl::gen;
use crate::repl::util;

pub fn clip(name: &str, database_path: &Path, key: &Key) -> Result<()> {
    let (_, entries) = util::read_database(database_path, key)?;
    clip_impl(name, &entries);
    Ok(())
}

fn clip_impl(
    name: &str,
    entries: &Entries,
) {
    if let Some(content) = entries.get(name) {
        if let Some(password) = gen::find_password(content) {
            copy_to_clipboard(password);
            println!("Password for {} copied to clipboard.", name);
        } else {
            println!("Error: no password for {} found.", name);
        }
    } else {
        println!("Error: no entry for {} found.", name);
    }
}

fn copy_to_clipboard(password: String) {
    let mut opts = Options::new();
    opts.serve_requests(ServeRequests::Only(1));
    opts.copy(Source::Bytes(password.into_bytes().into()), MimeType::Text).unwrap();
        // TODO: figure out why this doesn't compile with anyhow
        //.context("Unable to copy password to clipboard")
}

#[cfg(test)]
mod tests {
    use super::*;

    use std::io::Read;

    use wl_clipboard_rs::paste;
    use wl_clipboard_rs::paste::{ClipboardType, MimeType, Seat};

    const ENTRY: &str = "Entry";
    const VALUE: &str = "password: hello!";

    fn paste() -> String {
        let (mut pipe, _) = paste::get_contents(
            ClipboardType::Regular,
            Seat::Unspecified,
            MimeType::Text
        ).unwrap();
        let mut contents = Vec::new();
        pipe.read_to_end(&mut contents).unwrap();

        String::from_utf8(contents).unwrap()
    }

    #[test]
    fn password_of_entry_is_copied_to_clipboard() {
        let mut entries = Entries::new();
        entries.insert(ENTRY.to_string(), VALUE.to_string());

        clip_impl(ENTRY, &entries);

        let pasted = paste();
        assert_eq!("hello!", pasted);
    }
}
