// SPDX-License-Identifier: GPL-3.0-or-later

use std::path::Path;
use std::process::Command;

use anyhow::{bail, Context, Result};
use sodiumoxide::crypto::aead::xchacha20poly1305_ietf::Key;

use crate::entries::Entries;
use crate::repl::util;
use crate::tempfile::TempFile;

pub fn edit(name: &str, database_path: &Path, key: &Key) -> Result<()> {
    let (mut header, mut entries) = util::read_database(database_path, key)?;
    edit_impl(name, &mut entries, run_editor)?;
    util::write_database(database_path, &mut header, &entries, key)
}

fn edit_impl(
    name: &str,
    entries: &mut Entries,
    edit: fn(&str) -> Result<String>,
) -> Result<()> {
    if let Some(content) = entries.get_mut(name) {
        let new_content = edit(content)?;
        if new_content == *content {
            println!("Entry for {} unchanged.", name);
        } else {
            *content = new_content;
            println!("Edited entry for {}.", name);
        }
    } else {
        let content = edit("")?;
        entries.insert(name.to_string(), content);
        println!("New entry inserted for {}.", name);
    }
    Ok(())
}

fn run_editor(content: &str) -> Result<String> {
    let editor = std::env::var("EDITOR").context("EDITOR variable not set")?;

    let tempfile = TempFile::new();

    tempfile.write(content)?;

    let status = Command::new(&editor)
        .arg(&tempfile.path)
        .status()
        .with_context(|| format!("Unable to execute EDITOR {}", editor))?;

    if !status.success() {
        bail!("Execution of EDITOR {} failed", editor);
    }

    let new_content = tempfile.read()?;

    tempfile.remove()?;

    Ok(new_content.trim().to_string())
}

#[cfg(test)]
mod tests {
    use super::*;

    const ENTRY: &str = "Entry";
    const OLD_VALUE: &str = "Old Value";
    const NEW_VALUE: &str = "New Value";

    fn edit_stub(_content: &str) -> Result<String> {
        Ok(NEW_VALUE.to_string())
    }

    #[test]
    fn edit_overwrites_content_if_entry_exists() {
        let mut entries = Entries::new();
        entries.insert(ENTRY.to_string(), OLD_VALUE.to_string());

        edit_impl(ENTRY, &mut entries, edit_stub).unwrap();

        assert_eq!(entries.get(ENTRY).unwrap(), NEW_VALUE);
    }

    #[test]
    fn edit_inserts_entry_if_does_not_exist() {
        let mut entries = Entries::new();

        edit_impl(ENTRY, &mut entries, edit_stub).unwrap();

        assert_eq!(entries.get(ENTRY).unwrap(), NEW_VALUE);
    }
}
