// SPDX-License-Identifier: GPL-3.0-or-later

use std::io::BufRead;
use std::path::Path;

use anyhow::Result;
use sodiumoxide::crypto::aead::xchacha20poly1305_ietf::Key;

use crate::entries::Entries;
use crate::repl::util;

pub fn do_move(source_name: &str, dest_name: &str, database_path: &Path, key: &Key) -> Result<()> {
    let (mut header, mut entries) = util::read_database(database_path, key)?;
    do_move_impl(source_name, dest_name, &mut entries, &mut std::io::stdin().lock());
    util::write_database(database_path, &mut header, &entries, key)
}

fn do_move_impl(
    source_name: &str,
    dest_name: &str,
    entries: &mut Entries,
    input: &mut impl BufRead,
) {
    if !entries.contains_key(source_name) {
        println!("Error: entry for {} does not exist.", source_name);
        return;
    }

    if entries.contains_key(dest_name) {
        let prompt = format!("Entry for {} already exists. Overwrite it? [y/N] ", dest_name);
        let overwrite = util::confirm_prompt(&prompt, input);
        if !overwrite {
            println!("Entry for {} not overwritten.", dest_name);
            return;
        }
    }

    let content = entries.remove(source_name).unwrap();

    entries.insert(dest_name.to_string(), content);

    println!("Moved {} to {}.", source_name, dest_name);
}

#[cfg(test)]
mod tests {
    use super::*;

    const SOURCE: &str = "Source";
    const SOURCE_VALUE: &str = "Source Value";

    const DEST: &str = "Dest";
    const DEST_VALUE: &str = "Dest Value";

    #[test]
    fn dest_unmodified_if_source_does_not_exist() {
        let mut entries = Entries::new();
        entries.insert(DEST.to_string(), DEST_VALUE.to_string());

        do_move_impl(SOURCE, DEST, &mut entries, &mut "".as_bytes());

        assert_eq!(entries.get(DEST).unwrap(), DEST_VALUE);
    }

    #[test]
    fn entries_unmodified_if_dest_overwrite_cancelled() {
        let mut entries = Entries::new();

        entries.insert(SOURCE.to_string(), SOURCE_VALUE.to_string());
        entries.insert(DEST.to_string(), DEST_VALUE.to_string());

        let response = "n\n";
        do_move_impl(SOURCE, DEST, &mut entries, &mut response.as_bytes());

        assert_eq!(entries.get(SOURCE).unwrap(), SOURCE_VALUE);
        assert_eq!(entries.get(DEST).unwrap(), DEST_VALUE);
    }

    #[test]
    fn dest_overwritten_if_confirmed() {
        let mut entries = Entries::new();

        entries.insert(SOURCE.to_string(), SOURCE_VALUE.to_string());
        entries.insert(DEST.to_string(), DEST_VALUE.to_string());

        let response = "y\n";
        do_move_impl(SOURCE, DEST, &mut entries, &mut response.as_bytes());

        assert_eq!(entries.get(DEST).unwrap(), SOURCE_VALUE);
        assert!(!entries.contains_key(SOURCE));
    }

    #[test]
    fn source_moved_if_dest_does_not_exist() {
        let mut entries = Entries::new();

        entries.insert(SOURCE.to_string(), SOURCE_VALUE.to_string());

        do_move_impl(SOURCE, DEST, &mut entries, &mut "".as_bytes());

        assert_eq!(entries.get(DEST).unwrap(), SOURCE_VALUE);
        assert!(!entries.contains_key(SOURCE));
    }
}
