// SPDX-License-Identifier: GPL-3.0-or-later

use std::io::BufRead;
use std::path::Path;

use sodiumoxide::crypto::aead::xchacha20poly1305_ietf::Key;

use crate::entries::Entries;
use crate::repl::util;

use anyhow::Result;

pub fn del(name: &str, database_path: &Path, key: &Key) -> Result<()> {
    let (mut header, mut entries) = util::read_database(database_path, key)?;
    del_impl(name, &mut entries, &mut std::io::stdin().lock());
    util::write_database(database_path, &mut header, &entries, key)
}

fn del_impl(
    name: &str,
    entries: &mut Entries,
    input: &mut impl BufRead,
) {
    if entries.contains_key(name) {
        let prompt = format!("Are you sure you want to delete {}? [y/N] ", name);
        let delete = util::confirm_prompt(&prompt, input);
        if delete {
            entries.remove(name);
            println!("Entry for {} deleted.", name);
        } else {
            println!("Entry for {} not deleted.", name);
        }
    } else {
        println!("Error: entry for {} does not exist.", name);
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    const ENTRY: &str = "Entry";
    const VALUE: &str = "Value";

    #[test]
    fn entry_is_deleted_if_confirmed() {
        let mut entries = Entries::new();
        entries.insert(ENTRY.to_string(), VALUE.to_string());

        let response = "y\n";
        del_impl(ENTRY, &mut entries, &mut response.as_bytes());
        assert!(entries.is_empty());
    }

    #[test]
    fn entry_is_not_deleted_if_not_confirmed() {
        let mut entries = Entries::new();
        entries.insert(ENTRY.to_string(), VALUE.to_string());

        let response = "n\n";
        del_impl(ENTRY, &mut entries, &mut response.as_bytes());
        assert_eq!(entries.get(ENTRY).unwrap(), VALUE);
    }

    #[test]
    fn entries_unchanged_if_entry_name_does_not_exist() {
        let mut entries = Entries::new();
        entries.insert(ENTRY.to_string(), VALUE.to_string());

        del_impl("Does Not Exist", &mut entries, &mut "".as_bytes());
        assert_eq!(entries.get(ENTRY).unwrap(), VALUE);
    }
}
