// SPDX-License-Identifier: GPL-3.0-or-later

use std::convert::TryInto;

use sodiumoxide::randombytes;
use structopt::clap::arg_enum;

// TODO: implement this manually to get the right formatting...
arg_enum! {
    #[derive(Clone, Copy)]
    pub enum CharSet {
        Digit,
        Lower,
        Upper,
        Punct,
        Alpha,
        Alnum,
        Graph,
    }
}

impl std::fmt::Debug for CharSet {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        f.write_str(
            match self {
                CharSet::Digit => "digit",
                CharSet::Lower => "lower",
                CharSet::Upper => "upper",
                CharSet::Punct => "punct",
                CharSet::Alpha => "alpha",
                CharSet::Alnum => "alnum",
                CharSet::Graph => "graph",
            }
        )
        //match self {
            //CharSet::Digit => write!(f, "digit"),
            //CharSet::Lower => write!(f, "lower"),
            //CharSet::Upper => write!(f, "upper"),
            //CharSet::Punct => write!(f, "punct"),
            //CharSet::Alpha => write!(f, "alpha"),
            //CharSet::Alnum => write!(f, "alnum"),
            //CharSet::Graph => write!(f, "graph"),
        //}
    }
}

pub const DIGIT: &str = "0123456789";
pub const LOWER: &str = "abcdefghijklmnopqrstuvwxyz";
pub const UPPER: &str = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
pub const PUNCT: &str = "!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~";

// Hmm, how to reduce this duplication?
pub const ALPHA: &str = concat!("abcdefghijklmnopqrstuvwxyz", "ABCDEFGHIJKLMNOPQRSTUVWXYZ",);

pub const ALNUM: &str = concat!(
    "abcdefghijklmnopqrstuvwxyz",
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ",
    "0123456789",
);

pub const GRAPH: &str = concat!(
    "abcdefghijklmnopqrstuvwxyz",
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ",
    "0123456789",
    "!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~",
);

impl CharSet {
    fn chars(&self) -> &str {
        match self {
            CharSet::Digit => DIGIT,
            CharSet::Lower => LOWER,
            CharSet::Upper => UPPER,
            CharSet::Punct => PUNCT,
            CharSet::Alpha => ALPHA,
            CharSet::Alnum => ALNUM,
            CharSet::Graph => GRAPH,
        }
    }
}

pub fn random_string(charset: CharSet, length: u32) -> String {
    let chars = charset.chars();
    let mut password = String::with_capacity(length.try_into().unwrap());
    for _ in 0..length {
        let index: usize = randombytes::randombytes_uniform(chars.len().try_into().unwrap())
            .try_into()
            .unwrap();
        password.push(chars.chars().nth(index).unwrap());
    }

    password
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn random_string_length_matches_input_length() {
        let string = random_string(CharSet::Graph, 20);
        assert_eq!(string.len(), 20);
    }

    #[test]
    fn random_string_characters_are_taken_from_charset() {
        let charset = CharSet::Graph;
        let string = random_string(charset, 20);
        assert!(string.chars().all(|c| charset.chars().contains(c)))
    }
}
