// SPDX-License-Identifier: GPL-3.0-or-later

mod init;
mod key;
mod open;
mod pass;
mod passphrase;

use std::path::PathBuf;

use anyhow::Result;
use structopt::StructOpt;

#[derive(Debug, StructOpt)]
#[structopt(name = "napa", about = "A simple and secure command-line password manager")]
pub enum Cli {
    #[structopt(about = "Initialize a new database")]
    Init {
        #[structopt(parse(from_os_str))]
        database_path: PathBuf
    },
    #[structopt(about = "Open an existing database")]
    Open {
        #[structopt(parse(from_os_str))]
        database_path: PathBuf
    },
    #[structopt(about = "Change the passphrase of an existing database")]
    Pass {
        #[structopt(parse(from_os_str))]
        database_path: PathBuf
    }
}

impl Cli {
    pub fn run(&self) -> Result<()> {
        match self {
            Cli::Init { database_path } => init::init(database_path),
            Cli::Open { database_path } => open::open(database_path),
            Cli::Pass { database_path } => pass::pass(database_path),
        }
    }
}
