mod content_analysis_job;

use nNye_user_business::content_notification::ContentNotification;
use content_analysis_job::ContentAnalysisJob;
use std::io::Result;
use zmq::Context;
use zmq::{REQ, REP};
use nNye_solana_communication_layer::ISolana;
use std::rc::Rc;


/// This trait provides and abstraction for notifications queue.
pub trait INotificationsQueue {
    /// Adds a content notification to the queue.
    fn add_notification(&self, notification: &ContentNotification) -> Result<()>;

    /// Retrieves a notification from the queue.
    fn get_notification(&self) -> ContentNotification;
}

/// This structure implements INotificationsQueue.
pub struct NotificationsQueue {
    /// A zmq context
    context: Context,

    /// The endpoint of the queue's backend
    backend_endpoint: String,

    /// The endpoint of the queue's frontend
    frontend_endpoint: String,

    solana: Rc<dyn ISolana>
}

impl INotificationsQueue for NotificationsQueue {
    fn add_notification(&self, notification: &ContentNotification) -> Result<()> {
        let job = ContentAnalysisJob::from_business_entity(notification);
        let job_bytes = job.to_bytes();

        let requester = &self.context.socket(REQ).unwrap();
        requester
            .connect(&self.frontend_endpoint)
            .expect("failed to connect to requester");
        requester.send(job_bytes, 0).unwrap();
        println!("sent message to queue");
        Ok(())
    }

    fn get_notification(&self) -> ContentNotification {
        let responder = &self.context.socket(REP).unwrap();

        responder
            .connect(&self.backend_endpoint)
            .expect("failed to connect to responder");

        let job_bytes = responder.recv_bytes(0).unwrap();
        let job = ContentAnalysisJob::from_bytes(&job_bytes);
        job.to_business_entitiy(Rc::clone(&self.solana))
    }
}

impl NotificationsQueue {
    pub fn new(context: &Context, 
           frontend_endpoint: &String, 
           backend_endpoint: &String,
           solana: Rc<dyn ISolana>) -> NotificationsQueue {

        NotificationsQueue { 
            context: context.clone(), 
            frontend_endpoint: frontend_endpoint.clone(), 
            backend_endpoint: backend_endpoint.clone(),
            solana: solana
        } 
    }
}


