use bincode::{deserialize, serialize};
use serde::{Deserialize, Serialize};
use nNye_solana_communication_layer::ISolana;
use nNye_user_business::content_notification::ContentNotification;
use std::rc::Rc;

#[derive(Serialize, Deserialize)]
pub struct ContentAnalysisJob {
    pub pub_key: String,
    pub content_id: u32,
}

impl ContentAnalysisJob {

    //The two following methods are literally one-line wrappers around bincode, and therefore I'm
    //not going to test them.

    /// Serializes the job into a byte array.
    pub fn to_bytes(&self) -> Vec<u8> {
        serialize(&self).unwrap()
    }

    /// Deserializes a byte array into a job.
    pub fn from_bytes(bytes: &Vec<u8>) -> Self{
        let job: ContentAnalysisJob = deserialize(bytes).unwrap();
        job
    }

    /// Maps a business entity to a job.
    pub fn from_business_entity(notification: &ContentNotification) -> Self {
        ContentAnalysisJob {
            pub_key: notification.get_pub_key().clone(),
            content_id: notification.get_content_id().clone(),
        }
    }

    /// Maps a job to a business entity.
    pub fn to_business_entitiy(&self, solana: Rc<dyn ISolana>) -> ContentNotification {
        ContentNotification::new(self.pub_key.clone(), self.content_id.clone(), solana)
    }

}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn from_business_entity_successful() {
       //Arrange 
       let solana_mock = Rc::new(SolanaMock{});
       let notification = ContentNotification::new(String::from("some_key"), 1, solana_mock);

       //Act
       let job = ContentAnalysisJob::from_business_entity(&notification);

       //Assert
       let pub_key = job.pub_key;
       let content_id = job.content_id;
       assert!(String::from("some_key") == pub_key);
       assert!(1 == content_id);
    }

    #[test]
    fn to_business_entity_successful() {
        //Arrange 
        let job = ContentAnalysisJob{ pub_key: String::from("some_key"), content_id: 1 };
        let solana_mock = Rc::new(SolanaMock{});

        //Act
        let notification = job.to_business_entitiy(solana_mock);
        
        //Assert
        let pub_key = notification.get_pub_key();
        let content_id = notification.get_content_id();
        assert!("some_key" == pub_key);
        assert!(1 == content_id.clone());
    }

    struct SolanaMock {

    }

    impl ISolana for SolanaMock {
        fn get_ip(&self, pub_key: &String) -> String {
            String::from("1.2.3.4")
        }
    }
}

