use std::io::Result;
use std::io::Error;
use std::io::ErrorKind;
use diesel::prelude::*;
use diesel::sqlite::SqliteConnection;
use dotenv::dotenv;
use std::env;
use super::schema::prioritized_content::dsl::*;
use super::entities::prioritized_content::PrioritizedContentEntity;
use super::entities::prioritized_content::NewPrioritizedContentEntity;
use nNye_user_business::content::Content;
use nNye_user_business::enums::MediaType;

pub trait IPrioritizedContentRepository {
    fn add_prioritized_content(&self, content: &Content, _priority: &i32) -> Result<()>; //I need to put _ in front of the priority parameter since 'priority' is already an 
    fn get_top_prioritized_contents(&self, number_of_posts_to_retrieve: &i64);           //alias for the priority column.
    fn delete_top_prioritized_contents();
}

pub struct PrioritizedContentRepository {
    connection: SqliteConnection
}

impl IPrioritizedContentRepository for PrioritizedContentRepository {
    fn add_prioritized_content(&self, content: &Content, _priority: &i32) -> Result<()>{
        let entity = NewPrioritizedContentEntity::from_business(content, _priority);
        diesel::insert_into(prioritized_content)
               .values(&entity)
               .execute(&self.connection)
               .expect("Couldn't insert content");
        Ok(())
    }

    fn get_top_prioritized_contents(&self, number_of_posts_to_retrieve: &i64) {
        let prioritized_contents = prioritized_content
            .order_by(priority.asc())
            .limit(number_of_posts_to_retrieve.clone())
            .load::<PrioritizedContentEntity>(&self.connection)
            .expect("Some Error Occurred");
    }

    fn delete_top_prioritized_contents() {

    }

}

impl PrioritizedContentRepository {
    pub fn new() -> Self {
        dotenv().ok();
        let database_url = env::var("DATABASE_URL")
                               .expect("DATABASE_URL must be set.");
        let connection = SqliteConnection::establish(&database_url) 
                                          .expect(&format!("Couldn't connect to {}", &database_url));
        PrioritizedContentRepository{ connection: connection }
    }
}

pub fn resolve() -> impl IPrioritizedContentRepository {
   PrioritizedContentRepository::new() 
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn add_prioritized_content_successful() {
       //Arrange
       let content = Content::new(String::from("some_name"), 
                                  String::from("some_account"), 
                                  Some(String::from("some_text")), 
                                  Some(MediaType::PICTURE), 
                                  Some(String::from("some_path")), 
                                  String::from("some_date"));
       let _priority = 100;
        
       //Act
       let repository = resolve();
       let result = repository.add_prioritized_content(&content, &_priority);
       

      
    }
}
