use super::{
    configuration::Template, defaults, fs, parser::ArgumentTypes, types::FunctionArguments, utils,
    Path,
};

pub fn help_page(_args: FunctionArguments) {
    println!("Project Manager -- Help Page");
    println!(
        "      project_manager --list-projects   /--lp [language name]        - Lists all Project"
    );
    println!(
        "      project_manager --create-project  /--cp <name> <language name> - Creates a Project"
    );
    println!(
        "      project_manager --delete-project  /--dp <name> <language name> - Deletes a Project"
    );
    println!(
        "      project_manager --open-project    /--op <name> <language name> - Opens a Project"
    );
    println!(
        "      project_manager --list-templates  /--lt [language name]        - List all Templates"
    );
    println!(
        "      project_manager --create-template /--ct <template file path>   - Creates a Template"
    );
    println!(
        "      project_manager --get-template /--gt <template file path>      - Recives a Template"
    );
    println!(
        "      project_manager --delete-template /--dt <template name>        - Deletes a Template"
    );
    println!("Project Manager -- Help Page End")
}

pub fn list_projects(args: FunctionArguments) {
    let (ra, templates) = args;
    let list_args = utils::get_arguments("list_projects".to_string(), "lp".to_string(), ra);

    if let Some(list_argument) = list_args.clone() {
        if let ArgumentTypes::WithArguments(arguments) = list_argument.clone() {
            let choosed_templates = templates
                .iter()
                .filter(|temp| arguments.contains(&temp.language) || arguments.is_empty())
                .map(|temp| temp.clone())
                .collect::<Vec<Template>>();

            if choosed_templates.is_empty() {
                println!("No projects or no language found ");

                return;
            }

            for template in choosed_templates.clone() {
                let pr = utils::get_project_names(template.clone());

                println!("Projects in language {}:", template.language.clone());

                for name in pr {
                    println!("\t {}", name);
                }

                println!("");
            }

            return;
        }

        eprintln!("Argument Type is in the wrong format {:?}", list_argument);
        return;
    }

    eprintln!("Can not get Argument Type {:?}", list_args);
    return;
}
pub fn create_project(args: FunctionArguments) {
    let (ra, templates) = args;
    let create_args = utils::get_arguments("create_project".to_string(), "cp".to_string(), ra)
        .expect("Failed to get the arguments");

    if let ArgumentTypes::WithArguments(args) = create_args.clone() {
        if args.len() < 2 {
            println!("You need to specify the name and language");

            return;
        }

        let name = args.get(0).unwrap().clone();
        let language = args.get(1).unwrap().clone();

        let template = templates
            .iter()
            .find(|template| template.language.clone() == language)
            .expect("Failed to find the template")
            .clone();

        utils::create_project(name, template);

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        create_args
    )
}

pub fn delete_project(args: FunctionArguments) {
    let (ra, templates) = args;
    let delete_args = utils::get_arguments("delete_project".to_string(), "dp".to_string(), ra)
        .expect("Failed to get arguments");

    if let ArgumentTypes::WithArguments(args) = delete_args.clone() {
        if args.len() > 2 {
            eprintln!("You need to specify the project and the language");

            return;
        }

        let name = args.get(0).unwrap().clone();
        let language = args.get(1).unwrap().clone();

        let template = templates
            .iter()
            .find(|template| template.language.clone() == language)
            .expect("Failed to find the template")
            .clone();

        utils::delete_project(name, template);

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        delete_args
    )
}

pub fn open_project(args: FunctionArguments) {
    let (ra, templates) = args;
    let open_args = utils::get_arguments("open_project".to_string(), "op".to_string(), ra)
        .expect("Failed to get arguments");

    if let ArgumentTypes::WithArguments(args) = open_args.clone() {
        if args.len() > 2 {
            eprintln!("You need to specify the project and the language");

            return;
        }

        let name = args.get(0).unwrap().clone();
        let language = args.get(1).unwrap().clone();

        let template = templates
            .iter()
            .find(|template| template.language.clone() == language)
            .expect("Failed to find the template")
            .clone();

        utils::open_project(
            Path::new(&template.projects_dir)
                .join(format!("{}", name))
                .display()
                .to_string(),
            template.editor_open,
        );

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        open_args
    )
}

pub fn create_template(args: FunctionArguments) {
    let (ra, _) = args;
    let create_args = utils::get_arguments("create_template".to_string(), "ct".to_string(), ra);

    if let None = create_args {
        eprintln!("You need to specify the template file path");

        return;
    }

    if let ArgumentTypes::WithArguments(args) = create_args.clone().unwrap() {
        if args.is_empty() {
            eprintln!("You need to specify the template file path");

            return;
        }

        let path = Path::new(args.get(0).unwrap());

        let buffer = fs::read(path).expect("Failed to read the Path");
        let source = String::from_utf8(buffer).expect("Failed to convert buffer to string");
        let template = Template::from_toml(source).expect("Failed to string to template");

        utils::save_template(
            defaults::DEFAULT_TEMPLATE_PATH.to_string(),
            template.clone(),
        );

        println!(
            "Created and saved the template for the {} language",
            template.language
        );

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        create_args.unwrap().clone()
    )
}

pub fn list_templates(args: FunctionArguments) {
    let (_, templates) = args;

    println!("You have installed these templates:");

    for template in templates {
        println!(
            "\r Language: {} \n \t Projects in {} \n",
            template.language, template.projects_dir
        )
    }
}

pub fn get_template(args: FunctionArguments) {
    let (ra, templates) = args;
    let get_args = utils::get_arguments("get_args".to_string(), "gt".to_string(), ra);

    if let None = get_args {
        eprintln!("Can not get the arguments");
        return;
    }

    let args = get_args.unwrap();

    if let ArgumentTypes::WithArguments(args) = args {
        if args.is_empty() {
            println!("You need to specify the language");

            return;
        }

        let language_name = args.get(0).unwrap();
        let template = templates
            .iter()
            .find(|template| template.language.clone() == language_name.clone());

        if let None = template {
            eprintln!("The language {} does not exists", language_name);

            return;
        }

        let toml = template.unwrap().to_toml().unwrap();
        let name = format!("{}.toml", template.unwrap().language);

        fs::write(Path::new(&name), toml).expect_err("Failed to write the toml");

        println!(
            "Saved to current path, to update run project_manager --ct {}.toml",
            template.unwrap().language
        );

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        args
    )
}

pub fn delete_template(args: FunctionArguments) {
    let (ra, templates) = args;
    let delete_args = utils::get_arguments("delete_args".to_string(), "dt".to_string(), ra);

    if let None = delete_args {
        eprintln!("Can not get the arguments");
        return;
    }

    let arguments = delete_args.expect("");

    if let ArgumentTypes::WithArguments(args) = arguments.clone() {
        if args.is_empty() {
            eprintln!("You need to specify the language name")
        }

        let name = args.get(0).unwrap();
        let template_exists = templates
            .iter()
            .find(|template| template.language.clone() == name.clone())
            .is_none();

        if template_exists {
            eprintln!("The language {} does not exists", name);

            return;
        }

        utils::delete_template(defaults::DEFAULT_TEMPLATE_PATH.to_string(), name.clone());

        println!("The Language is deleted");

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        arguments
    )
}

pub fn create_dummy_template(args: FunctionArguments) {
    let (ra, _) = args;
    let dummy_args = utils::get_arguments(
        "create_dummy_template".to_string(),
        "cdt".to_string(),
        ra.clone(),
    );

    if let None = dummy_args {
        eprintln!("Can not get the arguments");
        return;
    }

    let arguments = dummy_args.expect("");

    if let ArgumentTypes::WithArguments(args) = arguments.clone() {
        if args.is_empty() {
            eprintln!("You need to specify the path");
            return;
        }

        let path = Path::new(args.get(0).expect("Failed to get the first index"));
        let template = Template::new(
            "".to_string(),
            "".to_string(),
            "".to_string(),
            Vec::new(),
            vec![(("//".to_string(), "./".to_string()))],
        );

        fs::write(
            path,
            template
                .to_toml()
                .expect("Failed to convert template to toml"),
        )
        .expect("Failed to write the file");

        return;
    }

    eprintln!(
        "Argument Parsing is wrong {:?} expected WithArguments",
        arguments
    )
}
